<?php

namespace April\Payments\Helper;

use April\Payments\Helper\Logger;
use April\Payments\Model\Config;
use Magento\Catalog\Model\ProductFactory;
use Magento\Catalog\Helper\ImageFactory as ImageHelper;

class Api
{
    const PROD_API           = 'https://api.au.meetapril.io';
    const SANDBOX_API        = 'https://api.sandbox.au.meetapril.io';
    const TST_API            = 'https://api.tst.au.meetapril.dev';
    const DEV_API            = 'https://api.dev.au.meetapril.dev';

    protected $config;
    protected $productFactory;
    protected $imageHelper;
    
    public function __construct(
        \April\Payments\Model\Config $config,
        ProductFactory $productFactory,
        ImageHelper $imageHelper
    ) {
        $this->config         = $config;
        $this->productFactory = $productFactory;
        $this->imageHelper    = $imageHelper;
    }

    public function createOrder( $orderParams )
    {
        $orderParams = [
            'CreateOnlineOrder' => $orderParams
        ];
        $resp = $this->request( $orderParams, 'orders' );
        return $resp;
    }

    public function orderPay( $payParams )
    {
        if (array_key_exists( 'threeDSAuthorisation', $payParams ) && !empty( $payParams['threeDSAuthorisation'] )) {
            $payData = [
                'Confirm3DSComplete' => [
                    'paymentTokenId' => $payParams['paymentToken'],
                    'threeDSResponse' => $payParams['threeDSAuthorisation']
                ]
            ];
        }else if(array_key_exists( 'paymentMethod', $payParams ) && array_key_exists( 'CardTokenDetails', $payParams['paymentMethod'] )) {
            $payData = [
                'PayByCard' => [
                    'paymentTokenId' => $payParams['paymentToken'],
                    'request3DS' => $payParams['request3DS']
                ]
            ];
        }else {
            $payData = [
                'PayByPaymentToken' => [
                    'paymentTokenId' => $payParams['paymentToken']
                ]
            ];
        }
        $resp = $this->request( $payData, 'orders/' . $payParams['merchantOrderId'] . '/pay' );
        return $resp;
    }

    public function refund( $refundParams )
    {
        $refundData = [
            'CreateRefund' => [
                'refundAmount' => $refundParams['amount']
            ]
        ];
        $resp = $this->request( $refundData, 'transactions/' . $refundParams['transactionId'], 'PATCH' );
        return $resp;
    }

    public function upsertCustomer( $custParams )
    {
        $custData = [
            'RegisterPluginConsumerCustomer' => [
                'referenceCustomerId' => $custParams['internalCustomerId'],
                'emailAddress' => $custParams['emailAddress']
            ]
        ];
        $resp = $this->request( $custData, 'customers/enterprise', 'POST' );
        return $resp;
    }

    public function signInCustomer( $custParams )
    {
        $custData = [
            'SignInConsumer' => [
                'customerId' => $custParams['customerId']
            ]
        ];
        $resp = $this->request( $custData, 'auth/signin', 'POST' );
        return $resp;
    }

    public function queryPaymentToken( $paymentToken )
    {
        $resp = $this->request( [], 'tokens/' . $paymentToken, 'GET' );
        return $resp;
    }

    protected function request( $params, $resource, $method = 'POST' )
    {
        //call your capture api here, incase of error throw exception.
        $secretKey = $this->config->getConfigData( 'aprilpayments', 'secretkey' );
        $dataString = json_encode( $params );

        $url = $this->getApiUrl();
        $url .= '/' . $resource;

        //open connection
        $ch = curl_init();
        curl_setopt( $ch, CURLOPT_URL, $url );
        curl_setopt( $ch, CURLOPT_CUSTOMREQUEST, $method );
        curl_setopt( $ch, CURLOPT_POSTFIELDS, $dataString );
        curl_setopt( $ch, CURLOPT_SSL_VERIFYPEER, true );
        curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
        curl_setopt( $ch, CURLOPT_FOLLOWLOCATION ,1 );
        curl_setopt( $ch, CURLOPT_CONNECTTIMEOUT, 120 ); // Timeout on connect (2 minutes)
        $httpHeader = [
            'Content-Type: application/json',
            'Content-Length: ' . strlen( $dataString )
        ];
        array_push( $httpHeader, 'Authorization: Bearer ' . $secretKey );
        curl_setopt( $ch, CURLOPT_HTTPHEADER, $httpHeader );

        $resp = array();

        \April\Payments\Helper\Logger::debug( 'April API called ' . $method . ' ' . $url );

        $result = curl_exec( $ch );

        if ( $result === false ) {
            $errorMsg = curl_error( $ch );
            $errorMsg = is_null( $errorMsg ) ? 'Unknown error occured while calling April API' : $errorMsg;
            $resp = [
              'data' => [ 'error' => 'curl_exec failed', 'message' => $errorMsg ]
            ];
        }else {
            
            $httpStatusCode = curl_getinfo( $ch, CURLINFO_HTTP_CODE );
            try {
                $data = json_decode( $result, true );
                $data = is_null( $data ) ? array() : $data;
                $resp = [
                  'httpStatusCode' => $httpStatusCode,
                  'data' => $data
                ];
                \April\Payments\Helper\Logger::debug( 'April API ' . $method . ' ' . $url . ' returned http status code ' . $httpStatusCode );
            } catch(Exception $e) {
                $resp = [
                  'httpStatusCode' => $httpStatusCode,
                  'error' => 'Failed to decode April API reponse'
                ];
            }
        }
        curl_close($ch);
        $this->logFailure( $resp, $method, $url );

        return $resp;
    }

    private function logFailure( $resp, $method, $url ) {
        $apiData = [
            'url' => $url,
            'method' => $method
        ];
        if ( key_exists( 'httpStatusCode', $resp ) ) {
            if ( $resp['httpStatusCode'] >= 500 ) {
                $logResp = array_merge( [
                    'error' => 'April API call failed due to server error'
                ], $apiData, $resp );
                \April\Payments\Helper\Logger::error( $logResp );
            } if ( $resp['httpStatusCode'] >= 300 ) {
                $logResp = array_merge( [
                    'error' => 'April API call failed due to request error'
                ], $apiData, $resp );
                \April\Payments\Helper\Logger::debug( $logResp );
            }
        } else {
            $logResp = array_merge( [
                'error' => 'Unable to execute April API call via curl_exec'
            ], $apiData, $resp );
            \April\Payments\Helper\Logger::error( $logResp );
        }
    }

    private function getApiUrl() {
        $env = $this->getEnv();
      	$url = self::PROD_API;

      	if ($env === 'dev') {
            $url = self::DEV_API;
        } else if ($env == 'tst') {
            $url = self::TST_API;
      	} else if ($env == 'sandbox') {
            $url = self::SANDBOX_API;
      	}
      	return $url;
    }

    private function getEnv() {
        $publishableKey = $this->config->getConfigData( 'aprilpayments', 'publishablekey' );
        $pkExp = explode( '_', $publishableKey );
        $env = $pkExp[0];
        return $env;
    }
}
