<?php
namespace April\Payments\Model;

use April\Payments\Helper\Logger;
use April\Payments\Model\Config;

class Transaction
{
    protected $_transactionId;
    protected $_merchantOrder;
    protected $_threeDSAuthorisation;
    protected $_paymentMethodCode;

    protected $api;
    protected $helper;
    protected $config;
    protected $threeDSAuthorisation;
    protected $merchantOrder;

    public function __construct(
        \April\Payments\Helper\Api $api,
        \April\Payments\Helper\Generic $helper,
        \April\Payments\Model\Config $config,
        \April\Payments\Model\ThreeDSAuthorisation $threeDSAuthorisation,
        \April\Payments\Model\MerchantOrder $merchantOrder
    ) {
    $this->api                      = $api;
        $this->helper               = $helper;
        $this->config               = $config;
        $this->threeDSAuthorisation = $threeDSAuthorisation;
        $this->merchantOrder        = $merchantOrder;
    }

    public function initOrder( $paymentMethodCode, $order, $threeDSAuthorisation = null )
    {
        $this->_paymentMethodCode = $paymentMethodCode;
        $this->threeDSAuthorisation->init( $paymentMethodCode );
        $this->merchantOrder->create( $order );

        if ( isset( $threeDSAuthorisation ) && !empty( $threeDSAuthorisation ) ) {
            $this->threeDSAuthorisation->process( $threeDSAuthorisation );
        }
    }

    public function create( $paymentToken, $request3DS )
    {
        $threeDSAuthorisation = null;
        if ( $this->threeDSAuthorisation->isThreeDSAuthorisationRequired() )
            $threeDSAuthorisation = $this->threeDSAuthorisation->_threeDSAuthorisation;

        $payParams = [
            'paymentToken' => $paymentToken,
            'request3DS' => $request3DS,
            'threeDSAuthorisation' => $threeDSAuthorisation,
            'merchantOrderId' => $this->merchantOrder->_merchantOrderId
        ];

        if ( is_null( $threeDSAuthorisation ) ) {
            $resp = $this->api->queryPaymentToken( $paymentToken );
            $paymentTokenData = $this->processQueryPaymentTokenAPIResponse( $resp );
            $payParams['paymentMethod'] = $paymentTokenData['method'];
        }

        $resp = $this->api->orderPay( $payParams );
        $transactionData = $this->processOrderPayAPIResponse( $resp );
        $this->_transactionId = $transactionData['transactionId'];
        $this->threeDSAuthorisation->clearThreeDSAuthorisation();
        return $transactionData;
    }

    public function refund(\Magento\Payment\Model\InfoInterface $payment, $amount )
    {
        $transactionId = $payment->getParentTransactionId();
        $amount = $this->getRefundAmountInOrderCurrency( $payment, $amount );

        $refundParams = [
            'transactionId' => $transactionId,
            'amount' => $this->helper->convertToCents( $amount )
        ];

        $refundResp = $this->api->refund( $refundParams );
        $this->processRefundAPIResponse( $refundResp );
    }

    public function getRefundAmountInOrderCurrency( \Magento\Payment\Model\InfoInterface $payment, $amount )
    {
        $order = $payment->getOrder();
        $creditmemo = $payment->getCreditmemo();

        if ($amount == $creditmemo->getBaseGrandTotal())
            $refundAmount = $creditmemo->getGrandTotal();
        else
            $refundAmount = $this->helper->convertBaseAmountToOrderCurrencyAmount( $amount, $order );

        return $refundAmount;
    }

    private function processQueryPaymentTokenAPIResponse( $resp )
    {
        $this->helper->isHttpSuccess($resp, 'Failed to fetch payment token data');
        return $resp['data'];
    }

    private function processOrderPayAPIResponse( $payResp )
    {
        $this->helper->isHttpSuccess($payResp, 'Failed to process the payment with April');
        if ( array_key_exists( 'PayOrderComplete', $payResp['data'] ) ) {
            return $payResp['data']['PayOrderComplete'];
        }else if ( array_key_exists( 'ThreeDSAuthorisationRequired', $payResp['data'] ) ) {
            if ( $this->helper->isAdmin() ) {
                $errorMsg = 'Card requires 3DS authorisation by customer.';
                \April\Payments\Helper\Logger::notice($errorMsg);
                throw new \Magento\Framework\Exception\LocalizedException( __( $errorMsg ) );
            }
            \April\Payments\Helper\Logger::debug('April initiating 3DS authorisation process');
            $this->handleThreeDSAuthorisationRequired( $payResp['data']['ThreeDSAuthorisationRequired'] );
        }
    }

    private function processRefundAPIResponse( $refundResp )
    {
        $this->helper->isHttpSuccess($refundResp, 'Failed to issue a refund via April');
    }

    private function handleThreeDSAuthorisationRequired( $threeDSAuthorisation )
    {
        $this->threeDSAuthorisation->setThreeDSAuthorisation( $threeDSAuthorisation );
        $errorMsg = '3DS Authorisation required::' . json_encode( $threeDSAuthorisation );
        throw new \Magento\Framework\Exception\CouldNotSaveException( __( $errorMsg ) );
    }
}
