define(
    [
        'jquery',
        'underscore',
        'Magento_Checkout/js/view/payment/default',
        'Magento_Checkout/js/action/select-payment-method',
        'Magento_Checkout/js/checkout-data',
        'Magento_Checkout/js/model/quote',
        'Magento_Ui/js/model/messages',
        'ko',
        'Magento_Checkout/js/model/full-screen-loader',
        'Magento_Checkout/js/model/payment/additional-validators',
        'Magento_Checkout/js/action/redirect-on-success',
    ],
    function (
        $,
        _,
        Component,
        selectPaymentMethodAction,
        checkoutData,
        quote,
        Messages,
        ko,
        fullScreenLoader,
        additionalValidators,
        redirectOnSuccessAction,
    ) {
        'use strict';

        return Component.extend({
            defaults: {
                template: 'April_Payments/payment/aprilpayments',
                aprilPaymentToken: null,
                aprilPaymentActionObject: null,
                aprilCheckoutAddress: {},
                aprilCheckoutAmount: {},
            },
            isInAction: ko.observable(true),

            /**
             * @returns {exports.initialize}
             */
             initialize: function () {
                 this._super();
                 this.observeAprilCheckoutParams();
                 this.updateAprilCheckoutAddress(quote.billingAddress());
                 this.updateAprilCheckoutAmount(quote.totals());

                 if (window.checkoutConfig.payment.aprilpayments.defaultPaymentMethod == '1') {
                    this.selectAprilPaymentMethod();
                 }

                 return this;
             },

             /**
              * Initialize child elements
              *
              * @returns {Component} Chainable.
              */
             initChildren: function () {
                 this.messageContainer = new Messages();
                 this.createMessagesComponent();
                 return this;
             },

             /**
              * Initialize query parameters object with billing address.
              * This will help to use the data from their default address of a logged in customer
              *
              * @returns {Component} Chainable.
              */
             observeAprilCheckoutParams: function() {
                console.log(quote.getOSC());
                quote.totals.subscribe(function () {
                   this.updateAprilCheckoutAmount(quote.totals());
                }, this);

                quote.shippingAddress.subscribe(function () {
                    if (!this.shouldNotUseShippingAddressUpdates()) {
                        this.updateAprilCheckoutAddress(quote.shippingAddress());
                    }
                }, this);

                quote.billingAddress.subscribe(function () {
                    if (!this.shouldNotUseBillingAddressUpdates()) {
                        this.updateAprilCheckoutAddress(quote.billingAddress());
                    }
                }, this);

                return this;
             },

             updateAprilCheckoutAddress: function(address) {
                var newAddress = _.pick(address, [
                  'firstname',
                  'middlename',
                  'lastname',
                  'street',
                  'city',
                  'region',
                  'postcode',
                  'countryId',
                  'telephone',
                ]);
                var changedAddress = this.addressChanged(newAddress, this.aprilCheckoutAddress);
                this.aprilCheckoutAddress = newAddress;
                if (changedAddress) {
                    this.renderAprilCheckout();
                }
             },

             addressChanged: function(addr1, addr2) {
               if (!addr1 && !addr2) {
                 return false;
               }
               if ((!addr1 && addr2) || (addr1 && !addr2)) {
                 return true;
               }
               if ((addr1.firstname != addr2.firstname)
                  || (addr1.lastname != addr2.lastname)
                  || (addr1.city != addr2.city)
                  || (addr1.region != addr2.region)
                  || (addr1.postcode != addr2.postcode)
                  || (addr1.countryId != addr2.countryId)
                  || (addr1.telephone != addr2.telephone)) {
                    return true;
                }
                return false;
             },

             updateAprilCheckoutAmount: function(amount) {
                 var newAmount = _.pick(amount, [
                   'grand_total',
                   'quote_currency_code'
                 ]);
                 var aprilCheckoutAmount = {
                    currencyCode: newAmount.quote_currency_code
                 };
                 var grand_segment = _.findWhere(amount.total_segments, {code: 'grand_total'});
                 if (grand_segment) {
                    aprilCheckoutAmount.grandTotal = Math.round(grand_segment.value * 100);
                 } else {
                    aprilCheckoutAmount.grandTotal = Math.round(newAmount.grand_total * 100);
                 }

                 var sameAmount = _.isEqual(aprilCheckoutAmount, this.aprilCheckoutAmount);
                 this.aprilCheckoutAmount = aprilCheckoutAmount;
                 if (!sameAmount) {
                     this.renderAprilCheckout();
                 }
             },

             getAprilCheckoutAmount: function() {
                return {
                   currency: this.aprilCheckoutAmount.currencyCode,
                   amount: this.aprilCheckoutAmount.grandTotal
                };
             },

             getAprilCheckoutAddress: function() {
                return this.aprilCheckoutAddress;
             },

             getResidentialAddress: function(address) {
                 var residAddress = (address.street ? address.street + ' ' : '');
                 residAddress += (address.city ? address.city + ' ' : '');
                 residAddress += (address.region ? address.region + ' ' : '');
                 residAddress += (address.postcode ? address.postcode + ' ' : '');
                 residAddress += (address.countryId ? address.countryId : '');

                 return residAddress;
             },

             showHideOptions: function() {
                 var hidePayLaterOption = false;
                 var hideFullPayOption = false;

                 if (this.getAvailablePaymentOption() === "paycard" || this.getCode() === "aprilpayments_paycard") {
                     hidePayLaterOption = true;
                 } else if (this.getAvailablePaymentOption() === "payplan" || this.getCode() === "aprilpayments_payplan") {
                     hideFullPayOption = true;
                 }
                 return {
                   hidePayLaterOption: hidePayLaterOption,
                   hideFullPayOption: hideFullPayOption,
                 }
             },

             renderAprilCheckout: function() {
                 var self = this;

                 if (!self.aprilCheckout) {
                    return;
                 }

                 var email = (quote.guestEmail && quote.guestEmail !== 'mail@example.com') ? quote.guestEmail : window.checkoutConfig.customerData.email;
                 var publicKey = window.checkoutConfig.payment.aprilpayments.publishablekey;
                 var primaryColor = window.checkoutConfig.payment.aprilpayments.primaryColor;
                 var customToken = window.checkoutConfig.payment.aprilpayments.customToken;
                 var preventWalletSubmit = window.checkoutConfig.payment.aprilpayments.preventWalletSubmit;

                 var payOptions = self.showHideOptions();

                 var address = self.getAprilCheckoutAddress();
                 var amount = self.getAprilCheckoutAmount();

                 var initParams = {
                   publicKey: publicKey,
                   email: email,
                   preventWalletSubmit: preventWalletSubmit,
                   customerFirstName: address.firstname,
                   customerMiddleName: address.middlename,
                   customerLastName: address.lastname,
                   customerResidentialAddress: self.getResidentialAddress(address),
                   phone: address.telephone,
                   hidePayLaterOption: payOptions.hidePayLaterOption,
                   hideFullPayOption: payOptions.hideFullPayOption,
                   paymentToken: self.handlePaymentToken.bind(self),
                   platform: 'magento',
                   platformVersion: '2',
                   platformPluginVersion: april.version,
                   customerToken: customToken
                 };

                 var renderParams = {
                     elementId: self.getPlaceHolderId(),
                     currency: amount.currency,
                     amount: amount.amount,
                     paymentType: self.getCode() == "aprilpayments" && self.isPayplanOptionPreselected() ? 'payplan' : 'paycard',
                     showPayNow: false,
                     showPayPlanSubmit: false,
                 };
                 if (primaryColor) {
                    renderParams.primaryColor = primaryColor;
                 }

                 self.aprilCheckout.init(initParams);
                 self.aprilCheckout.render(renderParams);
                 self.initAprilEventHandlers();
             },

            /**
            * In OSCs where billing address is placed after payment options,
            * iframe url should not be updated when user change/update/add their billing address
            *
            * 1. Swissup_Firecheckout OSC
            *
            * @returns {Boolean}.
            */
            shouldNotUseBillingAddressUpdates: function() {
                if (window.checkoutConfig.isFirecheckout || _.has(window.checkoutConfig, 'bssOsc') || quote.getOSC() === 'aheadworks') {
                   return true;
                }
                return false;
            },

            shouldNotUseShippingAddressUpdates: function() {

                if (_.has(window.checkoutConfig, 'bssOsc') || quote.getOSC() === 'aheadworks' || quote.isVirtual()) {
                   return true;
                }
                return false;
            },

            /**
             * Select payment method based on
             *  1. selectedPaymentOption - passed in configurations calculated based on sort_order
             *  2. isPayplanOptionPreselected - whether customer switched to BNPL in a widget
             */
            selectAprilPaymentMethod: function () {
                var selectedPaymentOption = window.checkoutConfig.payment.aprilpayments.selectedPaymentOption;
                var code = this.getCode();
                if ((code == "aprilpayments" && code === selectedPaymentOption) ||
                  (code == "aprilpayments_payplan" && (code === selectedPaymentOption || this.isPayplanOptionPreselected())) ||
                  (code == "aprilpayments_paycard" && code === selectedPaymentOption && !this.isPayplanOptionPreselected())) {

                    this.selectPaymentMethod();
                }
            },

            /**
             * Show text title
             * @returns {Boolean}
             */
            showTitleText: function () {
                var setting = window.checkoutConfig.payment.aprilpayments.titleSetting[this.getCode()];
                return (setting !== 'image_only');
            },

            /**
             * Show card image on title
             * @returns {Boolean}
             */
            showCardImage: function () {
                var setting = window.checkoutConfig.payment.aprilpayments.titleSetting[this.getCode()];
                return (setting !== 'text_only');
            },

            /**
             * Get instructions set in admin for the payment method.
             * @returns {String}
             */
            getInstructions: function () {
                return window.checkoutConfig.payment.aprilpayments.instructions[this.getCode()];
            },

            /**
             * Get available payment option value set by admin
             * @returns {String}
             */
            getAvailablePaymentOption: function () {
                return window.checkoutConfig.payment.aprilpayments.availablePaymentOption;
            },

            /**
             * Check if customer toggled to BNPL from a widget
             * @returns {Boolean}
             */
            isPayplanOptionPreselected: function () {
                return ($.cookie('aprilInstallmentToken')!== null && $.cookie('aprilInstallmentToken'));
            },

            /**
             * Get April iframe ID.
             * @returns {String}
             */
            getAprilInstanceId: function() {
                var code = this.getCode();
                return code + '_instance';
            },

            /**
             * Get customer's email set in checkout or logged in user email.
             * @returns {String}
             */
            getEmail: function () {
                if(quote.guestEmail) return quote.guestEmail;
                else return window.checkoutConfig.customerData.email;
            },

            /**
             * Get payment method data
             */
            getData: function () {
                return {
                    'method': this.item.method,
                    'po_number': null,
                    'additional_data': {
                        'april_payment_token': this.aprilPaymentToken,
                        'april_payment_action': this.aprilPaymentActionObject ? JSON.stringify(this.aprilPaymentActionObject) : null
                    }
                };
            },

            /**
             * Select April payment method by default
             * @return {Boolean}
             */
            selectPaymentMethod: function () {
                var code = this.getCode();
                selectPaymentMethodAction(this.getData());
                checkoutData.setSelectedPaymentMethod(this.item.method);
                return true;
            },

            /**
             * Hide loader when iframe is fully loaded.
             */
            onPlaceHolderRendered: function () {
                var self = this;
                var params = {
                  instanceId: self.getAprilInstanceId()
                };

                self.aprilPaymentToken = null;

                newAprilCheckout(params, function(err) {
                    if (err) {
                        console.log(err);
                        return;
                    }
                    self.aprilCheckout = april.getAprilCheckout(self.getAprilInstanceId());
                    self.renderAprilCheckout();
                });
            },

            initAprilEventHandlers: function() {
                var self = this;
                self.aprilCheckout.errorHandler(self.limepyCheckoutErrorHandler.bind(self));
                self.aprilCheckout.eventHandler(self.aprilCheckoutEventHandler.bind(self));
            },

            limepyCheckoutErrorHandler: function() {
                this.scrollIntoViewApril();
                fullScreenLoader.stopLoader();
            },

            aprilCheckoutEventHandler: function(aprilEvent) {
                if (aprilEvent.eventName == 'april_card_3DS_pending') {
                  this.scrollIntoViewApril();
                  fullScreenLoader.stopLoader();
                }
            },

            getPlaceHolderId: function() {
                return this.getCode() + '_placeholder';
            },

            handlePaymentToken: function(paymentToken) {
                var self = this;
                self.aprilPaymentToken = paymentToken;
                self.resetAprilPaymentActionObject();
                self.placeOrder();
            },

            /**
             * If the April payment token is not already available,
             * initiate submit process inside iframe.
             * @returns {Boolean}
             */
            requestPaymentToken: function() {
                if (this.aprilPaymentToken) {
                    return true;
                }
                this.showFullScreenLoader();

                this.aprilCheckout.submit();

                return false;
            },

            /**
             * Shows full screen loader
             * fullScreenLoader.startLoader() will not be used with BSS OSC
             */
            showFullScreenLoader: function() {
                if (_.isUndefined(window.checkoutConfig.bssOsc)) {
                    fullScreenLoader.startLoader();
                }
            },

            hideFullScreenLoader: function() {
                fullScreenLoader.stopLoader();
            },

            /**
             * Place order.
             *
             * @returns {Boolean}.
             */
             placeOrder: function (data, event) {
                 var self = this;

                 if (event) {
                     event.preventDefault();
                 }

                 var customErrorHandler = this.handlePlaceOrderErrors.bind(this);

                 self.messageContainer.clear();
                 if (this.validate() &&
                     additionalValidators.validate() &&
                     self.isPlaceOrderActionAllowed() === true
                 ) {
                     if (!this.requestPaymentToken()) {
                         return true;
                     }
                     self.isPlaceOrderActionAllowed(false);
                     self.getPlaceOrderDeferredObject()
                         .fail(customErrorHandler)
                         .done(
                             function () {
                                 self.afterPlaceOrder();

                                 if (self.redirectAfterPlaceOrder) {
                                     redirectOnSuccessAction.execute();
                                 }
                             }
                         );

                     return true;
                 }
                 return false;
             },

             handlePlaceOrderErrors: function (result) {
                 var self = this;
                 var status = result.status + " " + result.statusText;
                 var respMsg = result.responseJSON.message;

                 fullScreenLoader.stopLoader();
                 self.isPlaceOrderActionAllowed(true);
                 if (april.isThreeDSAuthorisationRequired(respMsg))
                 {
                     var paymentActionObject = self.getPaymentActionObject(respMsg);
                     self.openPayActionModal(paymentActionObject);
                 } else {
                    self.scrollIntoViewPaymentMethod();
                    self.resetAprilPaymentTokenAndData();
                 }
             },

             resetAprilPaymentTokenAndData: function() {
                this.aprilPaymentToken = null;
             },

             resetAprilPaymentActionObject: function() {
                this.aprilPaymentActionObject = null;
             },

             getPaymentActionObject: function( message ) {
                var payActParams = message.split('::');
                return JSON.parse(payActParams[1]);
             },

            openPayActionModal: function( paymentActionObject ) {
               var self = this;
               self.aprilCheckout.handleThreeDSAuthorisationRequired( paymentActionObject,
                   function() {
                      self.aprilPaymentActionObject = paymentActionObject;
                      self.isPlaceOrderActionAllowed(true);
                      self.placeOrder();
                   },
                   function(message) {
                     self.messageContainer.addErrorMessage({ 'message': message });
                     self.resetAprilPaymentTokenAndData();
                     self.resetAprilPaymentActionObject();
                     self.isPlaceOrderActionAllowed(true);
                   }
               );
         		},

            /**
             * If the April iframe if off screen,
             * scroll to April iframe
             */
            scrollIntoViewApril: function () {
                var code = this.getCode();
                var ifrElem = document.getElementById(this.getPlaceHolderId());
                ifrElem.scrollIntoView();
            },

            scrollIntoViewPaymentMethod: function () {
                var pMElem = $('#' + this.getPlaceHolderId()).parents('.payment-method-content')[0];
                pMElem.scrollIntoView();
            }
        });
    }
);
