<?php
namespace April\Payments\Helper;

use Magento\Backend\Model\Session;
use Magento\Framework\Validator\Exception;
use Magento\Store\Model\ScopeInterface;
use Magento\Authorization\Model\UserContextInterface;
use Magento\Framework\Exception\LocalizedException;

class Generic
{
    public $currentCustomer = null;
    protected $customerSession;
    protected $backendSessionQuote;
    protected $request;
    protected $appState;
    protected $storeManager;
    protected $order;
    protected $invoice;
    protected $creditmemo;
    protected $customerRegistry;
    protected $sessionManager;
    protected $customerCollection;
    protected $config;

    public function __construct(
        \Magento\Customer\Model\Session $customerSession,
        \Magento\Backend\Model\Session\Quote $backendSessionQuote,
        \Magento\Framework\App\Request\Http $request,
        \Magento\Framework\App\State $appState,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Sales\Model\Order $order,
        \Magento\Sales\Model\Order\Invoice $invoice,
        \Magento\Sales\Model\Order\Creditmemo $creditmemo,
        \Magento\Customer\Model\CustomerRegistry $customerRegistry,
        \Magento\Framework\Session\SessionManagerInterface $sessionManager,
        \April\Payments\Model\ResourceModel\AprilCustomer\Collection $customerCollection,
        \April\Payments\Model\Config $config
    ) {
        $this->customerSession = $customerSession;
        $this->backendSessionQuote = $backendSessionQuote;
        $this->request = $request;
        $this->appState = $appState;
        $this->storeManager = $storeManager;
        $this->order = $order;
        $this->invoice = $invoice;
        $this->creditmemo = $creditmemo;
        $this->customerRegistry = $customerRegistry;
        $this->sessionManager = $sessionManager;
        $this->customerCollection = $customerCollection;
        $this->config = $config;
    }

    public function getBackendSessionQuote()
    {
        return $this->backendSessionQuote->getQuote();
    }

    public function getStoreId()
    {
        if ($this->isAdmin())
        {
            if ($this->request->getParam('order_id', null))
            {
                $order = $this->order->load($this->request->getParam('order_id', null));
                return $order->getStoreId();
            }
            if ($this->request->getParam('invoice_id', null))
            {
                $invoice = $this->invoice->load($this->request->getParam('invoice_id', null));
                return $invoice->getStoreId();
            }
            else if ($this->request->getParam('creditmemo_id', null))
            {
                $creditmemo = $this->creditmemo->load($this->request->getParam('creditmemo_id', null));
                return $creditmemo->getStoreId();
            }
            else
            {
                $quote = $this->getBackendSessionQuote();
                return $quote->getStoreId();
            }
        }
        else
        {
            return $this->storeManager->getStore()->getId();
        }
    }

    public function convertToCents( $amount )
    {
        return round( $amount * 100 );
    }

    public function getOrderGrandTotal( $order )
    {
        return $this->convertToCents( $order->getGrandTotal() );
    }

    public function isAdmin()
    {
        $areaCode = $this->appState->getAreaCode();

        return $areaCode == \Magento\Backend\App\Area\FrontNameResolver::AREA_CODE;
    }

    public function getPublishableKey()
    {
        return $this->config->getConfigData('aprilpayments', 'publishablekey');
    }

    public function getCustomerModel()
    {
        $pk = $this->getPublishableKey();

        $customerId = $this->getCustomerId();
        $model = null;

        if (is_numeric($customerId) && $customerId > 0)
        {
            $model = $this->customerCollection->getByCustomerId($customerId, $pk);
            if ($model && $model->getId())
            {
                $model->updateSessionId();
                $this->currentCustomer = $model;
            }
        }

        if (!$this->currentCustomer)
            $this->currentCustomer = \Magento\Framework\App\ObjectManager::getInstance()->create('April\Payments\Model\AprilCustomer');

        return $this->currentCustomer;
    }

    public function isCustomerLoggedIn()
    {
        return $this->customerSession->isLoggedIn();
    }

    public function getCustomerId()
    {
        if ($this->customerSession->isLoggedIn())
        {
            return $this->customerSession->getCustomerId();
        }

        return null;
    }

    public function getMagentoCustomer()
    {
        if ($this->customerSession->getCustomer()->getEntityId())
            return $this->customerSession->getCustomer();

        $customerId = $this->getCustomerId();
        if (!$customerId) return;

        $customer = $this->customerRegistry->retrieve($customerId);

        if ($customer->getEntityId())
            return $customer;

        return null;
    }

    public function isGuest()
    {
        return !$this->customerSession->isLoggedIn();
    }

    public function isPaymentActionRequiredMessage($message)
    {
        return (strpos($message, "Payment Action required::") !== false);
    }

    public function convertBaseAmountToOrderCurrencyAmount( $amount, $order )
    {
        $rate = $order->getBaseToOrderRate();
        if ( empty( $rate ) )
            return $amount;

        return round( floatval( $amount * $rate), 4 );
    }

    public function isHttpSuccess($resp, $message = null) 
    {
        if ($resp['httpStatusCode'] >= 200 && $resp['httpStatusCode'] < 300) {
            return true;
        }
        $errorData = $this->getErrorMessage($resp, $message);

        switch ($errorData['logType']) {
            case 'notice':
                \April\Payments\Helper\Logger::notice([
                    'error' => $errorData['message'],
                    'data' => $resp
                ]);
            break;
            case 'error':
                \April\Payments\Helper\Logger::error([
                    'error' => $errorData['message'],
                    'data' => $resp
                ]);
            break;
            default:
                \April\Payments\Helper\Logger::log([
                    'error' => $errorData['message'],
                    'data' => $resp
                ]);
            break;
        }

        throw new \Magento\Framework\Exception\LocalizedException( __( $errorData['message'] ) );
    }

    private function getErrorMessage($resp, $altMessage)
    {
        $errorMsg = $altMessage;
        $logType = 'error';

        if ( array_key_exists( 'errorCode', $resp['data'] ) ) {
            $respData = $resp['data'];

            $apiErrorCode = $respData['errorCode'];
            $apiMessage = array_key_exists( 'message', $respData ) ? $respData['message'] : '';
            $apiDetail = array_key_exists( 'detail', $respData ) ? $respData['detail'] : '';
            $apiTracer = array_key_exists( 'tracer', $respData ) ? $respData['tracer'] : 'Not available';
    
            $errorList = array(
                'do_not_honor'          => 'Please contact your card issuer',
                'expired_card'          => 'Card expired',
                'fraudulent'            => 'Suspected fraudulent transaction',
                'incorrect_cvc'         => 'Incorrect CVC',
                'insufficient_funds'    => 'Insufficient funds',
                'invalid_cvc'           => 'Invalid CVC',
                'invalid_expiry_month'  => 'Invalid expiry month',
                'invalid_expiry_year'   => 'Invalid expiry year',
                'pickup_card'           => 'Card not allowed'
            );
            if (array_key_exists($apiErrorCode, $errorList)) {
                $errorMsg = $errorList[$apiErrorCode];
            } else {
                $errorMsg = $apiMessage . ' [' . $apiErrorCode . '] ' . $apiDetail . ' [Error reference: ' . $apiTracer . ']';
            }
            $logType = 'notice';
        }
        $errorMsg = !is_null( $errorMsg ) ? $errorMsg : 'Unknown error occured';

        return ['message' => $errorMsg, 'logType' => $logType];
    }
}
