<?php

namespace April\Payments\Model\Payment;

use April\Payments\Helper\Logger;
use Magento\Payment\Model\Method\Logger as MageLogger;
use Magento\Payment\Observer\AbstractDataAssignObserver;
use April\Payments\Model\Config;
use April\Payments\Helper\Generic;

class Aprilpayments extends \Magento\Payment\Model\Method\AbstractMethod
{

    protected $_code = "aprilpayments";

    protected $_canUseInternal              = true;
    protected $_canUseCheckout              = false;
    protected $_isGateway                   = true;
    protected $_canAuthorize                = true;
    protected $_canCapture                  = true;
    protected $_canCapturePartial           = true;
    protected $_canRefund                   = true;
    protected $_canRefundInvoicePartial     = true;

    protected $helper;
    protected $transaction;
    protected $config;

    const ACTION_AUTHORIZE = 'authorize';
    const ACTION_AUTHORIZE_CAPTURE = 'authorize_capture';

    protected $_formBlockType = \April\Payments\Block\Form::class;
    protected $_infoBlockType = \April\Payments\Block\Info::class;

    /**
     * @param \Magento\Framework\Model\Context $context
     * @param \Magento\Framework\Registry $registry
     * @param \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory
     * @param \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory
     * @param \Magento\Payment\Helper\Data $paymentData
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param MageLogger $mageLogger
     * @param \Magento\Framework\Model\ResourceModel\AbstractResource $resource
     * @param \Magento\Framework\Data\Collection\AbstractDb $resourceCollection
     * @param array $data
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        \April\Payments\Helper\Generic $helper,
        \April\Payments\Model\Config $config,
        \April\Payments\Model\Transaction $transaction,
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory,
        \Magento\Payment\Helper\Data $paymentData,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        MageLogger $mageLogger,
        ?\Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
        ?\Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null,
        array $data = []
    ) {
        $this->helper       = $helper;
        $this->transaction  = $transaction;
        $this->config       = $config;

        parent::__construct(
            $context,
            $registry,
            $extensionFactory,
            $customAttributeFactory,
            $paymentData,
            $scopeConfig,
            $mageLogger,
            $resource,
            $resourceCollection,
            $data
        );
        $this->initCanUseCheckout();
    }

    /**
     * Initialse this->_canUseCheckout value. Only "aprilpayments" method will be allowed in admin.
     *
     * @deprecated 100.2.0
     */
    private function initCanUseCheckout()
    {
        // If the publishable key is not set through admin, then make the April unavailable
        if (empty($this->config->getConfigData('aprilpayments', 'publishablekey'))) {
            $this->_canUseCheckout = false;
            return;
        }

        $indPayOptions = $this->config->getConfigData('aprilpayments', 'individual_payment_options');

        if ($this->_code === 'aprilpayments') {
            if ($indPayOptions == 0) {
                $this->_canUseCheckout = true;
            }
        } else {
            if ($indPayOptions == 1) {
                $availPayOptions = $this->config->getConfigData('aprilpayments', 'available_payment_options');
                if (($availPayOptions === "0") || ($this->_code === "aprilpayments_payplan" && $availPayOptions === "payplan") || ($this->_code === "aprilpayments_paycard" && $availPayOptions === "paycard")) {
                    $this->_canUseCheckout = true;
                }
            }
        }
    }

    /**
     * Retrieve information from payment configuration
     *
     * @param string $field
     * @param int|string|null|\Magento\Store\Model\Store $storeId
     *
     * @return mixed
     * @deprecated 100.2.0
     */
    public function getConfigData($field, $storeId = null)
    {
        $code = $this->_code;

        if ('order_place_redirect_url' === $field)
            return $this->getOrderPlaceRedirectUrl();

        return $this->config->getConfigData($code, $field, $storeId);
    }

    /**
     * Authorize payment abstract method
     *
     * @param \Magento\Framework\DataObject|InfoInterface $payment
     * @param float $amount
     * @return $this
     * @throws \Magento\Framework\Exception\LocalizedException
     * @api
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     * @deprecated 100.2.0
     */
    public function authorize(\Magento\Payment\Model\InfoInterface $payment, $amount)
    {
        parent::authorize($payment, $amount);
        $aprilPaymentToken = $payment->getAprilPaymentToken() ? $payment->getAprilPaymentToken() : $payment->getAdditionalInformation('april_payment_token');
        $errorMsg = null;
        if (! $aprilPaymentToken) {
            $errorMsg = 'Incorrect April payment token';
        }

        if ($errorMsg) {
            throw new \Magento\Framework\Exception\LocalizedException($errorMsg);
        }

        return $this;
    }

    /**
     * Payment capturing
     *
     * @param \Magento\Payment\Model\InfoInterface $payment
     * @param float $amount
     * @return $this
     * @throws \Magento\Framework\Validator\LocalizedException
     */
    public function capture(\Magento\Payment\Model\InfoInterface $payment, $amount)
    {
        $order = $payment->getOrder();
        \April\Payments\Helper\Logger::debug('April payment capture started for order');

        $paymentToken = $payment->getAprilPaymentToken() ? $payment->getAprilPaymentToken() : $payment->getAdditionalInformation('april_payment_token');
        $threeDSAuthorisation = $payment->getAdditionalInformation('april_payment_action');

        if (isset($threeDSAuthorisation) && !empty($threeDSAuthorisation)) {
            $threeDSAuthorisation = json_decode($threeDSAuthorisation);
        }

        // If it's a payment via admin area, then 3DS should not be requested
        $request3DS = $this->getRequest3DS($amount);

        if (!$paymentToken) {
            $errorMsg = 'April payment token missing';
            \April\Payments\Helper\Logger::error($errorMsg);
            throw new \Magento\Framework\Exception\LocalizedException(__($errorMsg));
        }

        // If secret key is not set in admin cifigurations, thro an error
        if (empty($this->config->getConfigData('aprilpayments', 'secretkey'))) {
            $errorMsg = 'April configurations error: Secret API key is not set';
            \April\Payments\Helper\Logger::error($errorMsg);
            throw new \Magento\Framework\Exception\LocalizedException(__($errorMsg));
        }

        // Init ThreeDSAuthorisation and MerchantOrder
        $this->transaction->initOrder($this->_code, $order, $threeDSAuthorisation);
        \April\Payments\Helper\Logger::debug('April order created for order');

        // Create new transaction
        $transactionData = $this->transaction->create($paymentToken, $request3DS);
        \April\Payments\Helper\Logger::debug('April transaction created for order');

        // Complete order
        $this->completeOrder($payment, $transactionData);
        \April\Payments\Helper\Logger::debug('April payment completed for order');

        return $this;
    }

    /**
     * Refund specified amount for payment
     *
     * @param \Magento\Framework\DataObject|InfoInterface $payment
     * @param float $amount
     * @return $this
     * @throws \Magento\Framework\Exception\LocalizedException
     * @api
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     * @deprecated 100.2.0
     */
    public function refund(\Magento\Payment\Model\InfoInterface $payment, $amount)
    {
        \April\Payments\Helper\Logger::debug('April refund initiated');
        $this->transaction->refund($payment, $amount);

        \April\Payments\Helper\Logger::debug('April refund success');
        return $this;
    }

    /**
     * Get config payment action url.
     *
     * Used to universalize payment actions when processing payment place.
     *
     * @return string
     * @api
     * @deprecated 100.2.0
     */
    public function getConfigPaymentAction()
    {
        return self::ACTION_AUTHORIZE_CAPTURE;
    }

    /**
     * Check whether payment method can be used
     *
     * @param \Magento\Quote\Api\Data\CartInterface|null $quote
     * @return bool
     * @deprecated 100.2.0
     */
    public function isAvailable(
        \Magento\Quote\Api\Data\CartInterface $quote = null
    ) {
        return parent::isAvailable($quote);
    }

    private function completeOrder($payment, $transactionData)
    {
        try {
            $payment->setTransactionId($transactionData['transactionId']);
            $payment->setIsTransactionClosed(1);
            // $additionalRawDetails = ['type' => $transactionData['type']];
            $payment->setAdditionalInformation('april_payment_type', $transactionData['payType']);
            $payment->setAdditionalInformation('april_transaction_id', $transactionData['transactionId']);
            // $payment->setTransactionAdditionalInfo(\Magento\Sales\Model\Order\Payment\Transaction::RAW_DETAILS, $additionalRawDetails);
        } catch (\Exception $e) {
            $errorMsg = 'April order completion error';
            \April\Payments\Helper\Logger::error([
                'error' => $errorMsg,
                'exception' => $e
            ]);
            throw new \Magento\Framework\Exception\LocalizedException(__($errorMsg));
        }
    }

    private function getRequest3DS($amount)
    {
        $request3DSSetting = $this->helper->isAdmin() ? false : boolval($this->config->getConfigData($this->_code, 'request_3ds'));
        $minAmt3DSSetting = $this->helper->isAdmin() ? 0 : floatval($this->config->getConfigData($this->_code, 'minimum_amount_3ds'));
        return ($request3DSSetting && $amount >= $minAmt3DSSetting);
    }
}
