<?php
namespace April\Payments\Model;

use April\Payments\Helper\Logger;
use April\Payments\Helper\Generic;
use April\Payments\Model\Config;
use Magento\Catalog\Model\ProductFactory;
use Magento\Catalog\Helper\ImageFactory as ImageHelper;

class MerchantOrder
{
    public $_orderId;
    public $_merchantOrderId;

    protected $api;
    protected $helper;
    protected $config;
    protected $session;
    protected $productFactory;
    protected $imageHelper;
    
    const SESSION_ORDER_PARAM_NAME = 'saved_order';

    public function __construct(
        \April\Payments\Helper\Api $api,
        \April\Payments\Helper\Generic $helper,
        \April\Payments\Model\Config $config,
        \Magento\Framework\Session\Generic $session,
        ProductFactory $productFactory,
        ImageHelper $imageHelper
    ) {
        $this->api            = $api;
        $this->helper         = $helper;
        $this->config         = $config;
        $this->session        = $session;
        $this->productFactory = $productFactory;
        $this->imageHelper    = $imageHelper;
    }

    public function create( $order )
    {
        $this->_orderId = $order->getIncrementId();
        $orderParams = $this->buildRequest( $order );
        $resp = $this->api->createOrder( $orderParams );
        $this->_merchantOrderId = $this->processAPIResponse( $resp );
    }

    private function processAPIResponse( $orderResp )
    {
        $this->helper->isHttpSuccess($orderResp, 'Failed to create the order with April');
        return $orderResp['data']['OnlineOrder']['order']['orderId'];
    }

    private function buildRequest( $order )
    {
        try {
            $total = $this->helper->getOrderGrandTotal( $order );
            $orderId = $this->_orderId;
            $currency = $order->getOrderCurrencyCode();
            $email = $order->getCustomerEmail();
            $items = [];

            $imageHelperObj = $this->imageHelper->create();
            $productLoader = $this->productFactory->create();

            foreach ( $order->getAllItems() as $item ) {
                if( $item->getData( 'has_children' ) ) {
                    continue;
                }
                $product = $productLoader->load( $item->getProductId() );
                $items[] = array(
                    'amount'         => array(
                        'minorCurrencyUnits' => $this->helper->convertToCents( $product->getFinalPrice() ),
                        'currency' => $currency
                    ),
                    'description'    => strval( $item->getName() ),
                    'sku'            => strval( $item->getSku() ),
                    'quantity'       => $item->getQtyOrdered(),
                    'imageUrl'       => $imageHelperObj
                                        ->init( $product, 'product_small_image' )
                                        ->setImageFile( $product->getImage() )
                                        ->getUrl(),
                );
            }

            /* In case of virtual/downloadable/gift card products, get details from billing address */
            $shippingAddressArray = [];
            if ( $order->getShippingAddress() ) {
                $shippingAddressObject = $order->getShippingAddress();
                /* Get shipping address street lines */
                $shippingAddressStreetLine1 = null;
                $shippingAddressStreetLine2 = null;
                if ( is_array( $shippingAddressObject->getStreet() ) ) {
                    $shippingAddressStreetLine1 = array_key_exists( 0, $shippingAddressObject->getStreet() ) ? $shippingAddressObject->getStreet()[0] : null;
                    $shippingAddressStreetLine2 = array_key_exists( 1, $shippingAddressObject->getStreet() ) ? $shippingAddressObject->getStreet()[1] : null;
                } else {
                    $shippingAddressStreetLine1 = $shippingAddressObject->getStreet();
                }

                $shippingAddressArray = [
                    'amount' => $this->helper->convertToCents( $order->getShippingAmount() ),
                    'address' => [
                        'line1' => strval( $shippingAddressStreetLine1 ),
                        'line2' => strval( $shippingAddressStreetLine2 ),
                        'city' => strval( $shippingAddressObject->getCity() ),
                        'state' => strval( $shippingAddressObject->getRegion() ),
                        'postalCode' => strval( $shippingAddressObject->getPostcode() ),
                        'country' => strval( $shippingAddressObject->getCountryId() )
                    ],
                    'carrier' => strval( $order->getShippingDescription() ),
                    'name' => strval( $shippingAddressObject->getName() ),
                    'phoneNumber' => strval( $shippingAddressObject->getTelephone() )
                ];
            }

            /* Get billing address street lines */
            $billingAddressStreetLine1 = null;
            $billingAddressStreetLine2 = null;
            if ( is_array( $order->getBillingAddress()->getStreet() ) ) {
                $billingAddressStreetLine1 = array_key_exists( 0, $order->getBillingAddress()->getStreet() ) ? $order->getBillingAddress()->getStreet()[0] : null;
                $billingAddressStreetLine2 = array_key_exists( 1, $order->getBillingAddress()->getStreet() ) ? $order->getBillingAddress()->getStreet()[1] : null;
            } else {
                $billingAddressStreetLine1 = $order->getBillingAddress()->getStreet();
            }

            $description = ( count( $items ) > 0 ? strval( count( $items ) ) : 'No' ) . ' item(s)';
            $orderParams = [
                'referenceOrderId' => strval( $orderId ),
                'amount'   => [ 'minorCurrencyUnits' => $total, 'currency' => $currency ],
                'customerEmailAddress'   => $email ? $email : '',
                'customerPhoneNumber' => strval( $order->getBillingAddress()->getTelephone() ),
                'description'     => $description,
                'items'           => $items,
                'discountAmount'  =>  $this->helper->convertToCents( $order->getDiscountAmount() * -1 ),
                'billing' => [
                    'address' => [
                        'line1' => strval( $billingAddressStreetLine1 ),
                        'line2' => strval( $billingAddressStreetLine2 ),
                        'city' => strval( $order->getBillingAddress()->getCity() ),
                        'state' => strval( $order->getBillingAddress()->getRegion() ),
                        'postalCode' => strval( $order->getBillingAddress()->getPostcode() ),
                        'country' => strval( $order->getBillingAddress()->getCountryId() )
                    ],
                    'name' => strval( $order->getBillingAddress()->getName() ),
                    'phoneNumber' => strval( $order->getBillingAddress()->getTelephone() ),
                ]
            ];
            if ( count( $shippingAddressArray ) ) {
                $orderParams['shipping'] = $shippingAddressArray;
            }

            return $orderParams;

        } catch ( \Exception $e ) {
            $msg = $e.getMessage();
            $errorMsg = 'April API error: Error occured when generating order creation payload.';
            throw new \Magento\Framework\Exception\LocalizedException( __( $errorMsg ) );
        }
    }
}
