<?php
/**
 * Copyright © April Solutions.
 * This file is part of April payments module for PrestaShop.
 *
 * @author    April Solutions (https://www.meetapril.com/)
 * @copyright April Solutions
 * @license   https://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 */

use PrestaShop\PrestaShop\Core\Payment\PaymentOption;

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once(_PS_MODULE_DIR_ . 'april/autoload.php');

class April extends PaymentModule
{
    private $_html = '';
    private $_postErrors = [];

    public function __construct()
    {
        $this->name = 'april';
        $this->tab = 'payments_gateways';
        $this->version = '1.0.2';
        $this->author = 'April';
        $this->controllers = ['payment', 'redirect'];

        $this->limited_countries = ['AU', 'NZ'];
        $this->limited_currencies = ['AUD', 'NZD'];

        $config = Configuration::getMultiple( [
          'APRIL_TITLE',
          'APRIL_REQUEST_3DS',
          'APRIL_MINIMUM_AMOUNT_3DS',
          'APRIL_PUBLIC_API_KEY',
          'APRIL_SECRET_API_KEY',
          'APRIL_PRIMARY_COLOR',
          'APRIL_WALLET_PAYMENTS_PLACE_ORDER',
          'APRIL_ENABLED'
        ] );

        if ( isset( $config['APRIL_PUBLIC_API_KEY'] )) {
            $this->publicAPIKey = $config['APRIL_PUBLIC_API_KEY'];
        }
        if ( isset( $config['APRIL_SECRET_API_KEY'] )) {
            $this->secretAPIKey = $config['APRIL_SECRET_API_KEY'];
        }
        if ( isset( $config['APRIL_ENABLED'] )) {
            $this->aprilEnabled = $config['APRIL_ENABLED'];
        }
        $this->paymentMethodTitle = $config['APRIL_TITLE'];
        $this->aprilPrimaryColor = $config['APRIL_PRIMARY_COLOR'];
        $this->preventWalletSubmit = !$config['APRIL_WALLET_PAYMENTS_PLACE_ORDER'];

        $this->aprilApi = new AprilApi( $this->secretAPIKey );

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = $this->l( 'April Payments' );
        $this->description = $this->l( 'April payments integration.' );
        $this->confirmUninstall = $this->l( 'Are you sure you want to delete these details?' );
        $this->ps_versions_compliancy = ['min' => '1.7.1.0', 'max' => _PS_VERSION_];

        if ( !isset( $this->publicAPIKey ) || !isset( $this->secretAPIKey ) || empty( $this->publicAPIKey ) || empty( $this->secretAPIKey ) ) {
            $this->warning = $this->l( 'The "APRIL_PUBLIC_API_KEY" and "APRIL_SECRET_API_KEY" fields must be configured before using this module.' );
        }
        if ( !count( Currency::checkPaymentCurrencies( $this->id ) ) ) {
            $this->warning = $this->trans('No currency has been set for this module.', [], 'Modules.Checkpayment.Admin');
        }
    }

    public function install()
    {
        include( dirname( __FILE__ ) . '/sql/install.php' );

        return parent::install()
            && $this->registerHook( 'displayHeader' )
            && $this->registerHook( 'paymentOptions' )
            && $this->registerHook( 'displayPaymentReturn' )
            && $this->registerHook( 'actionOrderSlipAdd' )
            && $this->registerHook( 'actionOrderStatusUpdate' );

            // Disabled widget related hooks
            // && $this->registerHook('displayProductPriceBlock')
            // && $this->registerHook('displayExpressCheckout')
            // && $this->registerHook('displayShoppingCartFooter');
    }

    public function uninstall()
    {
        include( dirname( __FILE__ ) . '/sql/uninstall.php' );

        return Configuration::deleteByName( 'APRIL_ENABLED' )
            && Configuration::deleteByName( 'APRIL_TITLE' )
            && Configuration::deleteByName( 'APRIL_REQUEST_3DS' )
            && Configuration::deleteByName( 'APRIL_MINIMUM_AMOUNT_3DS' )
            && Configuration::deleteByName( 'APRIL_PUBLIC_API_KEY' )
            && Configuration::deleteByName( 'APRIL_SECRET_API_KEY' )
            && Configuration::deleteByName( 'APRIL_WALLET_PAYMENTS_PLACE_ORDER' )
            && Configuration::deleteByName( 'APRIL_PRIMARY_COLOR' )
            && parent::uninstall();
    }

    private function _postValidation()
    {
        if ( Tools::isSubmit( 'btnSubmit' ) ) {
            if ( Tools::getValue( 'APRIL_ENABLED' ) === '1' ) {
                if ( !Tools::getValue( 'APRIL_PUBLIC_API_KEY' ) ) {
                    $this->_postErrors[] = $this->trans( 'The "APRIL_PUBLIC_API_KEY" field is required.', [], 'Modules.Checkpayment.Admin' );
                } elseif (!Tools::getValue('APRIL_SECRET_API_KEY') && empty( Configuration::get( 'APRIL_SECRET_API_KEY' ) ) ) {
                    $this->_postErrors[] = $this->trans( 'The "APRIL_SECRET_API_KEY" field is required.', [], 'Modules.Checkpayment.Admin' );
                }
            }
        }
    }

    private function _postProcess()
    {
        if ( Tools::isSubmit('btnSubmit') ) {
            $postSecretKey = Tools::getValue( 'APRIL_SECRET_API_KEY' );
            if ( empty( $postSecretKey ) ) {
                $postSecretKey = Configuration::get( 'APRIL_SECRET_API_KEY' );
            }

            Configuration::updateValue( 'APRIL_ENABLED', Tools::getValue( 'APRIL_ENABLED' ) );
            Configuration::updateValue( 'APRIL_TITLE', Tools::getValue( 'APRIL_TITLE' ) );
            Configuration::updateValue( 'APRIL_REQUEST_3DS', Tools::getValue( 'APRIL_REQUEST_3DS' ) );
            Configuration::updateValue( 'APRIL_MINIMUM_AMOUNT_3DS', Tools::getValue( 'APRIL_MINIMUM_AMOUNT_3DS' ) );
            Configuration::updateValue( 'APRIL_PUBLIC_API_KEY', Tools::getValue( 'APRIL_PUBLIC_API_KEY' ) );
            Configuration::updateValue( 'APRIL_SECRET_API_KEY', $postSecretKey );
            Configuration::updateValue( 'APRIL_PRIMARY_COLOR', Tools::getValue( 'APRIL_PRIMARY_COLOR' ) );
            Configuration::updateValue( 'APRIL_WALLET_PAYMENTS_PLACE_ORDER', Tools::getValue( 'APRIL_WALLET_PAYMENTS_PLACE_ORDER' ) );
        }
        $this->_html .= $this->displayConfirmation( $this->trans( 'Settings updated', [], 'Admin.Notifications.Success' ) );
    }

    private function _displayCheck()
    {
        return $this->display( __FILE__, './views/templates/hook/infos.tpl' );
    }

    public function getContent()
    {
        $this->_html = '';

        if ( Tools::isSubmit( 'btnSubmit' ) ) {
            $this->_postValidation();
            if ( !count( $this->_postErrors ) ) {
                $this->_postProcess();
            } else {
                foreach ( $this->_postErrors as $err ) {
                    $this->_html .= $this->displayError( $err );
                }
            }
        }

        $this->_html .= $this->_displayCheck();
        $this->_html .= $this->renderForm();

        return $this->_html;
    }

    public function hookPaymentOptions( $params )
    {
        if ( !$this->active || !$this->aprilEnabled ) {
            return;
        }
        if ( !$this->checkCurrency( $params['cart'] )) {
            return;
        }

        $this->smarty->assign(
            $this->getTemplateVars()
        );

        $newOption = new PaymentOption();

        $newOption->setModuleName( $this->name )
                ->setCallToActionText( $this->trans( $this->paymentMethodTitle, [], 'Modules.Checkpayment.Admin' ) )
                ->setAction( $this->context->link->getModuleLink( $this->name, 'payment', [], true ) )
                ->setAdditionalInformation( $this->fetch( 'module:april/views/templates/front/payment_infos.tpl' ) );

        return [$newOption];
    }

    /**
     * Payment method selection page header.
     * Registers April js files
     *
     * @param array $params
     * @return string|void
     */
    public function hookDisplayHeader( $params )
    {
        $controller = $this->context->controller;
        if ( $controller instanceof OrderController || $controller instanceof OrderOpcController ) {
            if ( isset( $this->context->cookie->aprilErrors ) ) {
                // Process errors from other pages.
                $controller->errors = array_merge(
                    $controller->errors,
                    explode( "\n", $this->context->cookie->aprilErrors )
                );
                unset( $this->context->cookie->aprilErrors );

                // Unset HTTP_REFERER from global server variable to avoid back link display in error message.
                $_SERVER['HTTP_REFERER'] = null;
                $this->context->smarty->assign( 'server', $_SERVER );
            }

            $this->context->controller->registerJavascript(
                'april-core',
                'modules/' . $this->name . '/views/js/april.js',
                [
                  'position' => 'bottom',
                  'priority' => 80,
                ]
            );

            $this->context->controller->registerJavascript(
                'april-checkout',
                'modules/' . $this->name . '/views/js/april-checkout.js',
                [
                  'position' => 'bottom',
                  'priority' => 100,
                ]
            );

            $html = '';
            $html .= '<script src="' . 'https://checkout-v3.au.meetapril.io/v3/checkout-v3.0.0.min.js' . '"></script>';

            return $html;
        }
    }

    public function hookDisplayPaymentReturn( $params )
    {
        if ( !$this->active || !$this->aprilEnabled ) {
            return;
        }

        return $this->fetch( 'module:april/views/templates/hook/payment_return.tpl' );
    }

    /**
     * Handles partial refunds.
     * Partial refund amount will be refunded via April.
     *
     * @param array $params
     * @return boolean
     */
    public function hookActionOrderSlipAdd( $params )
    {
        $order = $params['order'];

        if ( $order->module === $this->name ) {
            if ( !Validate::isLoadedObject( $order ) ) {
                return false;
            }

            $orderPayment = $this->_getOrderPaymentFromOrder( $order );
            if ( empty( $orderPayment ) || empty( $orderPayment->transaction_id ) ) {
                return true;
            }
            $transactionId = $orderPayment->transaction_id;
            $currency = new Currency( (int)$orderPayment->id_currency );
            $currencyCode = $currency->iso_code;

            $allParams = Tools::getAllValues();
            $refundAmountCal = new AprilRefundAmountCalculator();
            $refundAmount = $refundAmountCal->calculate( array_merge( $params, $allParams ) );

            PrestaShopLogger::addLog(
                json_encode( $refundData ),
                1,
                AprilHelper::E_CODE_PARTIAL_REFUND,
                "AprilRefund",
                1
            );
            $resp = $this->aprilApi->refund( $transactionId, AprilHelper::convertToCents( $refundAmount ) );
            if ( !$resp->isSuccess() ) {
                return false;
            }
        }

        return true;
    }

    /**
     * Handles the event when order state is updated to Canceled or Refund.
     * Total/Balance amount of the transaction will be refunded via April.
     *
     * @param array $params
     * @return boolean
     */
    public function hookActionOrderStatusUpdate( &$params )
    {
        $order = new Order( (int)$params['id_order'] );

        if ( Validate::isLoadedObject( $order ) && $order->module === $this->name )
        {
            $orderPayment = $this->_getOrderPaymentFromOrder( $order );
            $transactionId = $orderPayment->transaction_id;

            if ( !empty( $transactionId ) ) {
                $statusList = array(
                    (int) Configuration::get('PS_OS_REFUND'),
                    (int) Configuration::get('PS_OS_CANCELED'),
                );
                if ( in_array( (int)$params['newOrderStatus']->id, $statusList ) ) {

                    PrestaShopLogger::addLog(
                        'Full refund: ' . (string) $transactionId,
                        1,
                        AprilHelper::E_CODE_FULL_REFUND,
                        "AprilRefund",
                        1
                    );
                    $resp = $this->aprilApi->fullRefund( $transactionId );
                    if ( !$resp->isSuccess() ) {
                        return false;
                    }
                }
            }
        }

        return true;
    }

    // TODO: This hook is currently not registered
    public function hookDisplayExpressCheckout( $params )
    {
        // TODO: Check configuration setting before displaying the widget
        $total = $this->context->cart->getOrderTotal();
        $this->context->smarty->assign( "april", $this->_getInstallmentsDisplay( $total ) );

        return '<div><p style="margin-top: 15px;">4 installments of ' . $this->_getInstallmentsDisplay( $total ) . '</p></div>';
    }

    // TODO: This hook is currently not registered
    public function hookDisplayProductPriceBlock( $params )
    {
        // TODO: Check configuration setting before displaying the widget
        $current_controller = Tools::getValue('controller');

        if ( $current_controller == "product" &&
            $params["type"] == "after_price" )
        {
            $this->context->smarty->assign( "april", $this->_getInstallmentsDisplay( $params["product"]["price_amount"] ) );
            return $this->context->smarty->fetch("module:april/views/templates/front/product_page.tpl");
        }
    }

    // TODO: This hook is currently not registered
    public function hookDisplayShoppingCartFooter( $params )
    {
        // TODO: Check configuration setting before displaying the widget
        $total = $this->context->cart->getOrderTotal();
        $this->context->smarty->assign( "april", $this->_getInstallmentsDisplay( $total ) );

        return null ;
    }

    public function checkCurrency( $cart )
    {
        $currency_order = new Currency( (int) ( $cart->id_currency ) );
        $currencies_module = $this->getCurrency( (int) $cart->id_currency );

        if ( is_array( $currencies_module ) ) {
            foreach ( $currencies_module as $currency_module ) {
                if ($currency_order->id == $currency_module['id_currency']) {
                    return true;
                }
            }
        }

        return false;
    }

    public function renderForm()
    {
        $fields_form = [
            'form' => [
                'legend' => [
                    'title' => $this->l( 'April Config Details' ),
                    'icon' => 'icon-envelope',
                ],
                'input' => [
                    [
                        'type' => 'switch',
                        'label' => $this->l( 'Enable' ),
                        'name' => 'APRIL_ENABLED',
                        'is_bool' => true,
                        'values' => [
                            [
                                'id' => 'active_on',
                                'value' => true,
                                'label' => $this->l( 'Enabled' )
                            ],
                            [
                                'id' => 'active_off',
                                'value' => false,
                                'label' => $this->l( 'Disabled' )
                            ]
                        ],
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->l( 'Title' ),
                        'name' => 'APRIL_TITLE',
                        'size' => 50,
                        'required' => true,
                    ],
                    [
                        'type' => 'switch',
                        'label' => $this->l( 'Request 3DS on Payments' ),
                        'name' => 'APRIL_REQUEST_3DS',
                        'is_bool' => true,
                        'values' => [
                            [
                                'id' => 'active_on',
                                'value' => true,
                                'label' => $this->l( 'Enabled' )
                            ],
                            [
                                'id' => 'active_off',
                                'value' => false,
                                'label' => $this->l( 'Disabled' )
                            ]
                        ],
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->l( 'Minimum Amount for 3DS' ),
                        'name' => 'APRIL_MINIMUM_AMOUNT_3DS',
                        'size' => 10,
                        'desc' => $this->l( '3DS will not be requested for payments below this amount.' ),
                        'required' => false,
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->l( 'April Public API Key' ),
                        'name' => 'APRIL_PUBLIC_API_KEY',
                        'size' => 30,
                        'required' => true,
                    ],
                    [
                        'type' => 'password',
                        'label' => $this->l( 'April Secret API Key' ),
                        'name' => 'APRIL_SECRET_API_KEY',
                        'size' => 30,
                        'required' => true,
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->l( 'Primary color (hex code)' ),
                        'name' => 'APRIL_PRIMARY_COLOR',
                        'size' => 10,
                        'required' => false,
                    ],
                    [
                        'type' => 'switch',
                        'label' => $this->l( 'Allow Wallet payments to place the order' ),
                        'desc' => $this->l( 'Submit orders immediately when a digital wallet payment such as Apple Pay or Google Pay is selected.' ),
                        'name' => 'APRIL_WALLET_PAYMENTS_PLACE_ORDER',
                        'is_bool' => true,
                        'values' => [
                            [
                                'id' => 'active_on',
                                'value' => true,
                                'label' => $this->l( 'Enabled' )
                            ],
                            [
                                'id' => 'active_off',
                                'value' => false,
                                'label' => $this->l( 'Disabled' )
                            ]
                        ],
                    ],
                ],
                'submit' => [
                    'title' => $this->l('Save'),
                ],
            ],
        ];

        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->id = (int) Tools::getValue( 'id_carrier' );
        $helper->identifier = $this->identifier;
        $helper->submit_action = 'btnSubmit';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false) . '&configure=' . $this->name . '&tab_module=' . $this->tab . '&module_name=' . $this->name;
        $helper->token = Tools::getAdminTokenLite( 'AdminModules' );
        $helper->tpl_vars = [
            'fields_value' => $this->getConfigFieldsValues(),
        ];

        return $helper->generateForm( [$fields_form] );
    }

    public function getConfigFieldsValues()
    {
        $title = Configuration::get( 'APRIL_TITLE' );
        if ( empty( $title ) ) {
            $title ='Split or full payments';
        }
        return [
            'APRIL_ENABLED' => Tools::getValue( 'APRIL_ENABLED', Configuration::get( 'APRIL_ENABLED' ) ),
            'APRIL_TITLE' => Tools::getValue( 'APRIL_TITLE', $title ),
            'APRIL_REQUEST_3DS' => Tools::getValue( 'APRIL_REQUEST_3DS', Configuration::get( 'APRIL_REQUEST_3DS' ) ),
            'APRIL_MINIMUM_AMOUNT_3DS' => Tools::getValue( 'APRIL_MINIMUM_AMOUNT_3DS', Configuration::get( 'APRIL_MINIMUM_AMOUNT_3DS' ) ),
            'APRIL_PUBLIC_API_KEY' => Tools::getValue( 'APRIL_PUBLIC_API_KEY', Configuration::get( 'APRIL_PUBLIC_API_KEY' ) ),
            'APRIL_SECRET_API_KEY' => Tools::getValue( 'APRIL_SECRET_API_KEY', Configuration::get( 'APRIL_SECRET_API_KEY' ) ),
            'APRIL_PRIMARY_COLOR' => Tools::getValue( 'APRIL_PRIMARY_COLOR', Configuration::get( 'APRIL_PRIMARY_COLOR' ) ),
            'APRIL_WALLET_PAYMENTS_PLACE_ORDER' => Tools::getValue( 'APRIL_WALLET_PAYMENTS_PLACE_ORDER', Configuration::get( 'APRIL_WALLET_PAYMENTS_PLACE_ORDER' ) )
        ];
    }

    public function getTemplateVars()
    {
        $errorCookie = $this->context->cookie->aprilErrors;
        $cart = $this->context->cart;
        $customer = $this->context->customer;

        /**
         * TODO: $aprilMerchantId should be obtained via April API.
         */
        $aprilMerchantId = 'lp_merchant_id';
        $aprilCustomer = new AprilCustomer();
        $aprilCustomerToken = null;

        if ( $customer->isLogged() ) {
            $aprilCustomer = $aprilCustomer->getCustomerById( $customer->id, $aprilMerchantId );
            $aprilPluginCustomer = new AprilPluginCustomer();

            if (empty($aprilCustomer->id) === true) {
                $aprilPluginCustomer->internalCustomerId = $customer->id;
                $aprilPluginCustomer->firstName = $customer->firstname;
                $aprilPluginCustomer->lastName = $customer->lastname;
                $aprilPluginCustomer->email = $customer->email;
                $aprilPluginCustomer->create();

                $aprilCustomer->id_customer = $customer->id;
                $aprilCustomer->april_customer_id = $aprilPluginCustomer->customerId;
                $aprilCustomer->april_merchant_id = $aprilMerchantId;
                $aprilCustomer->save();
            }
            $aprilCustomerToken = $aprilPluginCustomer->signInCustomer( $aprilCustomer->april_customer_id );
        }


        $email = $customer->email;

        $checkoutConfig = array();

        $checkoutConfig['email'] = $email;
        $checkoutConfig['publishablekey'] = $this->publicAPIKey;
        $checkoutConfig['primarycolor'] = $this->aprilPrimaryColor;
        $checkoutConfig['preventwalletsubmit'] = $this->preventWalletSubmit;
        $checkoutConfig['amount'] = AprilHelper::convertToCents( $cart->getOrderTotal( true, Cart::BOTH ) );
        $checkoutConfig['currency'] = Currency::getIsoCodeById( (int) $cart->id_currency );
        $checkoutConfig['address'] = $this->_prepareBillingAddress();
        $checkoutConfig['customertoken'] = $aprilCustomerToken;

        $this->smarty->assign( 'aprilErrors', $errorCookie );
        $this->smarty->assign( 'checkoutConfig', json_encode( $checkoutConfig ) );
    }

    private function _prepareBillingAddress() {
        $billing_address = new Address( $this->context->cart->id_address_invoice );

        $country_object = new Country( $billing_address->id_country );
        $state_object = new State( $billing_address->id_state );

        $address = array();
        $address['firstname'] = $billing_address->firstname;
        $address['lastname'] = $billing_address->lastname;
        $address['street'] = $billing_address->address1 . (!empty( $billing_address->address2 ) ? ' ' . $billing_address->address2 : '');
        $address['postcode'] = $billing_address->postcode;
        $address['city'] =  $billing_address->city;
        $address['countryId'] =  $country_object->iso_code;
        $address['telephone'] = $billing_address->phone;

        if( !empty( $billing_address->id_state ) && !empty( $state_object ) ) {
            $address['region']   =   $state_object->iso_code;
        }

        return $address;
    }

    private function _getOrderPaymentFromOrder( $order )
    {
        $payment = $order->getOrderPaymentCollection();
        if ( isset( $payment[0] ) )
        {
            return $payment[0];
        }
        return null;
    }

    private function _getInstallmentsDisplay( $amount = NULL )
    {
        if( empty( $amount ) ) {
            $cart = $this->context->cart;
            $amount = $cart->getOrderTotal();
        }
        $instalment = round( $amount / 4, 2, PHP_ROUND_HALF_UP );

        return '$' . number_format( $instalment, 2, '.', ',' );
    }

    public static function getPrecision()
    {
        if ( version_compare( _PS_VERSION_, '1.7.7', '<' ) ) {
            return _PS_PRICE_DISPLAY_PRECISION_;
        } else {
            return Context::getContext()->getComputingPrecision();
        }
    }
}
