<?php
/**
 * Copyright © April Solutions.
 * This file is part of April payments module for PrestaShop.
 *
 * @author    April Solutions (https://www.meetapril.com/)
 * @copyright April Solutions
 * @license   https://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 */

class AprilOrder
{
    public $orderId;
    public $internalOrderId = 'prestashop_temp_order_id';
    public $billing = [];
    public $shipping = [];
    public $products = [];
    public $orderTotal = 0;
    public $currencyCode;

    private $aprilApi;

    public function __construct()
    {
        $this->aprilApi = new AprilApi( Configuration::get( 'APRIL_SECRET_API_KEY' ) );
    }

    public function create()
    {
        $orderId = 'prestashop_temp_order_id';
        $orderData = [
            'CreateOnlineOrder' => [
                'referenceOrderId' => (string) $orderId,
                'amount' => array( 'minorCurrencyUnits' => AprilHelper::convertToCents( $this->orderTotal ), 'currency' => $this->currencyCode ),
                'description' => 'Prestashop order',
                'items' => $this->products,
                'billing' => $this->billing
            ]
        ];
        if ( isset( $this->shipping ) && !empty( $this->shipping ) ) {
            $orderData['shipping'] = $this->shipping;
        }

        $resp = $this->aprilApi->createOrder( $orderData );
        $respData = $resp->getData( 'OnlineOrder' );

        if( !$resp->isSuccess() || is_null( $respData ) ) {
            PrestaShopLogger::addLog(
                json_encode( $resp->getErrorData() ),
                3,
                AprilHelper::E_CODE_APRIL_ORDER_CREATION_FAILED,
                'AprilOrder',
                1
            );
            throw new Exception( 'Failed to create order with April' );
        }
        
        $this->orderId = $respData['order']['orderId'];
    }

    public function setInternalOrderId( $aprilOrderId, $internalOrderId )
    {
        $orderData = [
            'UpdateReferenceOrderId' => [
                'referenceOrderId' => (string) $internalOrderId
            ]
        ];

        try {
            $resp = $this->aprilApi->updateOrder( $aprilOrderId, $orderData );
            if( !$resp->isSuccess() ) {
                PrestaShopLogger::addLog(
                    json_encode( $resp->getErrorData() ),
                    3,
                    AprilHelper::E_CODE_INTL_ORDER_UPDATE_FAILED,
                    'AprilOrder',
                    1
                );
            }
        } catch ( Exception $e ) {
            PrestaShopLogger::addLog(
                'Error message: ' . (string) $e->getMessage(),
                3,
                AprilHelper::E_CODE_INTL_ORDER_UPDATE_FAILED,
                'AprilOrder',
                1
            );
        }
    }
}
