<?php
/**
 * Copyright © April Solutions.
 * This file is part of April payments module for PrestaShop.
 *
 * @author    April Solutions (https://www.meetapril.com/)
 * @copyright April Solutions
 * @license   https://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 */

abstract class AprilAbstractModuleFrontController extends ModuleFrontController
{
    public function postProcess()
    {

        try {
            $success = ( $this->validations() );
            if ( !$success ) {
                return;
            }
            $success = ( $this->additionalValidations() );
            if ( !$success ) {
                return;
            }

            $this->processPayment();

        } catch( AprilRedirectException $e ) {

            PrestaShopLogger::addLog(
                'Redirect URL: ' . (string) $e->getUrl(),
                1,
                AprilHelper::E_CODE_URL_REDIRECT,
                'AprilAbstractController',
                1
            );
            Tools::redirect( $e->getUrl() );

        } catch( Exception $e ) {
            $message = $e->getMessage();
            $this->_paymentProcessingError( $message, 'index.php?controller=order&step=4' );
        }
    }

    protected function validations() {
        $cart = $this->context->cart;
        if ( $cart->id_customer == 0 || $cart->id_address_delivery == 0 || $cart->id_address_invoice == 0 || !$this->module->active ) {
            Tools::redirect('index.php?controller=order&step=1');
            return false;
        }

        if ( !( $this->module instanceof April ) ) {
            $this->_paymentProcessingError( 'Incorrect payment method', 'index.php?controller=order' );
            return false;
        }

        // Check that this payment option is still available in case the customer changed his address just before the end of the checkout process
        $authorized = false;
        foreach ( Module::getPaymentModules() as $module ) {
            if ( $module['name'] == 'april' ) {
                $authorized = true;
                break;
            }
        }

        if ( !$authorized ) {
            die($this->trans( 'This payment method is not available.', [], 'Modules.Checkpayment.Shop' ) );
            return false;
        }

        $customer = new Customer( $cart->id_customer );
        if ( !Validate::isLoadedObject( $customer ) ) {
            $this->_paymentProcessingError( 'Customer error', 'index.php?controller=order&step=1');
            return false;
        }

        return true;
    }

    protected function completeOrder( $transactionId, $aprilOrderId )
    {
        if ( empty( $transactionId ) || is_null( $transactionId ) ) {
            throw new Exception( 'April payments failed. Transaction ID not available.' );
        }
        $cart = $this->context->cart;
        $orderTotal = $this->getOrderGrandTotal();
        $orderCurrency = $this->getOrderCurrency();
        $customer = new Customer( $cart->id_customer );


        $this->module->validateOrder(
            (int) $cart->id,
            (int) Configuration::get( 'PS_OS_PAYMENT' ),
            $orderTotal,
            $this->module->displayName,
            null,
            array( 'transaction_id' => $transactionId ),
            (int) $orderCurrency->id,
            false,
            $customer->secure_key
        );

        $orderId = $this->module->currentOrder;
        $aprilOrder = new AprilOrder();
        $aprilOrder->setInternalOrderId( $aprilOrderId, $orderId );

        Tools::redirect('index.php?controller=order-confirmation&id_cart=' . (int) $cart->id
          . '&id_module=' . (int) $this->module->id
          . '&id_order=' . $orderId
          . '&key=' . $customer->secure_key
        );
    }

    protected function getOrderGrandTotal()
    {
        $cart = $this->context->cart;
        return (float) $cart->getOrderTotal( true, Cart::BOTH );
    }

    protected function getOrderCurrency()
    {
        return $this->context->currency;
    }

    protected function _paymentProcessingError( $message, $redirect = null ) {

        PrestaShopLogger::addLog(
            json_encode([
                'message' => $message,
                'redirect' => $redirect
            ]),
            3,
            AprilHelper::E_CODE_PAYMENT_PROCESS_FAILED,
            'AprilAbstractController',
            1
        );

        $this->context->cookie->aprilErrors = $this->module->l( $message, 'submit' );
        if ( $redirect ) {
            Tools::redirect( $redirect );
        }
    }

    /** @return  boolean*/
    abstract protected function additionalValidations();

    /** @return  void*/
    abstract protected function processPayment();

}
