<?php
/**
 * Copyright © April Solutions.
 * This file is part of April payments module for PrestaShop.
 *
 * @author    April Solutions (https://www.meetapril.com/)
 * @copyright April Solutions
 * @license   https://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 */

use Symfony\Component\HttpClient\HttpClient;

class AprilApi
{
    private $secretKey = null;

    const APRIL_PROD_API    = 'https://api.au.meetapril.io';
    const APRIL_SANDBOX_API = 'https://api.sandbox.au.meetapril.io';
    const APRIL_TST_API     = 'https://api.tst.au.meetapril.dev';
    const APRIL_DEV_API     = 'https://api.dev.au.meetapril.dev';

    public function __construct( $secretKey )
    {
        $this->secretKey = $secretKey;
    }

    public function createOrder( $params )
    {
        return $this->request( $params, 'orders' );
    }

    public function updateOrder( $orderId, $orderParams )
    {
        return $this->request( $orderParams, 'orders/' . $orderId, 'PATCH' );
    }

    public function orderPay( $orderId, $params )
    {
        return $this->request( $params, 'orders/' . $orderId . '/pay' );
    }

    public function queryPaymentToken( $paymentToken )
    {
        return $this->request( [], 'tokens/' . $paymentToken, 'GET' );
    }

    public function refund( $transactionId, $amount )
    {
        $refundData = [
            'CreateRefund' => [
              'refundAmount' => $amount,
            ]
        ];
        return $this->request( $refundData, 'transactions/' . (string)$transactionId, 'PATCH' );
    }

    public function fullRefund( $transactionId )
    {
        $txnResp = $this->request( null, 'transactions/' . (string)$transactionId, 'GET' );
        if ( !$txnResp->isSuccess() ) {
            return $txnResp;
        }
        $txnData = $txnResp->getData();
        $refundableAmount = (int)$txnData['purchaseDetails']['amount']['minorCurrencyUnits'] - (int)$txnData['purchaseDetails']['refundedAmount'];

        if ( $refundableAmount <= 0 ) {
            return $txnResp;
        }

        return $this->refund( $transactionId, $refundableAmount );
    }

    public function upsertCustomer( $custParams )
    {
        return $this->request( $custParams, 'customers/enterprise', 'POST' );
    }

    public function signInCustomer( $custParams )
    {
        $custData = [
            'customerId' => $custParams['customerId']
        ];
        return $this->request( $custParams, 'auth/signin', 'POST' );
    }

    protected function request( $params, $resource, $method = 'POST' )
    {
        //call your capture api here, incase of error throw exception.
        $dataString = json_encode( $params );

        $url = $this->getApiUrl();
        $url .= '/' . $resource;

        //open connection
        $ch = curl_init();
        curl_setopt( $ch, CURLOPT_URL, $url );
        curl_setopt( $ch, CURLOPT_CUSTOMREQUEST, $method );
        if ($method !== 'GET') {
          curl_setopt( $ch, CURLOPT_POSTFIELDS, $dataString );
        }
        curl_setopt( $ch, CURLOPT_SSL_VERIFYPEER, true );
        curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
        curl_setopt( $ch, CURLOPT_FOLLOWLOCATION ,1 );
        curl_setopt( $ch, CURLOPT_CONNECTTIMEOUT, 120 ); // Timeout on connect (2 minutes)
        $httpHeader = [
            'Content-Type: application/json',
        ];
        if ($method !== 'GET') {
            array_push( $httpHeader, 'Content-Length: ' . strlen( $dataString ) );
        }
        array_push( $httpHeader, 'Authorization: Bearer ' . $this->secretKey );

        curl_setopt( $ch, CURLOPT_HTTPHEADER, $httpHeader );

        $resp = new AprilApiResponse();
        $result = curl_exec( $ch );

        PrestaShopLogger::addLog(
            json_encode([
                'method' => $method,
                'url' => $url
            ]),
            1,
            AprilHelper::E_CODE_API_REQUEST,
            'AprilApi',
            1
        );

        if ( $result === false ) {
            $errorMsg = curl_error( $ch );
            $errorMsg = is_null( $errorMsg ) ? 'Unknown error occured while calling April API' : $errorMsg;

            $resp = new AprilApiResponse( null, [], [ 'message' => $errorMsg ] );
            PrestaShopLogger::addLog(
                'CURL result was empty',
                3,
                AprilHelper::E_CODE_API_REQUEST_FAILED,
                'AprilApi',
                1
            );

        }else {
            $httpStatusCode = curl_getinfo( $ch, CURLINFO_HTTP_CODE );
            try {
                $data = json_decode( $result, true );
                $data = is_null( $data ) ? [] : $data;
                if ( $httpStatusCode >= 200 && $httpStatusCode < 300 ) {
                    $resp = new AprilApiResponse( $httpStatusCode, $data );
                } else {
                    $resp = new AprilApiResponse( $httpStatusCode, [], $data );
                }

                PrestaShopLogger::addLog(
                    json_encode([
                        'httpStatusCode' => $httpStatusCode
                    ]),
                    1,
                    AprilHelper::E_CODE_API_SUCCESS_RESPONSE,
                    'AprilApi',
                    1
                );
            } catch( Exception $e ) {
                PrestaShopLogger::addLog(
                    'Error message: ' . (string) $e->getMessage(),
                    3,
                    AprilHelper::E_CODE_API_REQUEST_FAILED,
                    'AprilApi',
                    1
                );
                $resp = new AprilApiResponse( $httpStatusCode, [], [ 'message' => 'Unable to process April API response' ] );
            }
        }
        curl_close($ch);

        return $resp;
    }

    private function getApiUrl() {
        $env = $this->getEnv();
        $url = self::APRIL_DEV_API;

        if ($env === 'live') {
            $url = self::APRIL_PROD_API;
        } else if ($env == 'test' || $env == 'sandbox') {
            $url = self::APRIL_SANDBOX_API;
        } else if ($env == 'tst') {
            $url = self::APRIL_TST_API;
        }
        return $url;
    }

    private function getEnv() {
        $pkExp = explode( '_', $this->secretKey );
        $env = $pkExp[0];
        return $env;
    }
}
