<?php
/**
 * Copyright © April Solutions.
 * This file is part of April payments module for PrestaShop.
 *
 * @author    April Solutions (https://www.meetapril.com/)
 * @copyright April Solutions
 * @license   https://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 */

class AprilTransaction
{
    public $transactionId;
    public $aprilOrderId;
    public $amount = 0;
    public $currencyCode;

    private $aprilApi;

    public function __construct()
    {
        $this->aprilApi = new AprilApi( Configuration::get( 'APRIL_SECRET_API_KEY' ) );
    }

    public function create( $aprilOrderId, $amount, $currencyCode, $aprilPaymentToken, $aprilPaymentData, $aprilPaymentAction, $request3DS = false, $paymentActionRequired = null ) {

        $this->aprilOrderId = $aprilOrderId;
        $this->amount = $amount;
        $this->currencyCode = $currencyCode;

        if ( !empty( $paymentActionRequired )) {
            $payOrderParams = [
                'Confirm3DSComplete' => [
                    'paymentTokenId' => $aprilPaymentToken,
                    'threeDSResponse' => $paymentActionRequired
                ]
            ];
        } else if ( !empty( $aprilPaymentData ) &&  array_key_exists('CardTokenDetails', $aprilPaymentData) ) {
            $payOrderParams = [
                'PayByCard' => [
                    'paymentTokenId' => $aprilPaymentToken,
                    'request3DS' => $request3DS,
                    'returnURL' => Tools::getHttpHost(true) . __PS_BASE_URI__ . 'module/april/redirect'
                ]
            ];
        }else {
            $payOrderParams = [
                'PayByPaymentToken' => [
                    'paymentTokenId' => $aprilPaymentToken
                ]
            ];
        }
        
        $resp = $this->aprilApi->orderPay( $aprilOrderId, $payOrderParams );

        if ( $resp->isSuccess() ) {
            $payOrderComplete = $resp->getData( 'PayOrderComplete' );
            if ( !is_null( $payOrderComplete ) ) {
                $this->transactionId = $payOrderComplete['transactionId'];
                
                $aprilPaymentAction->reset();
                $aprilPaymentAction->save();

                PrestaShopLogger::addLog(
                    json_encode( $payOrderComplete ),
                    1,
                    AprilHelper::E_CODE_APRIL_SUCCESS_PAYMENT,
                    'AprilTransaction',
                    1
                );
                return;
            }

            $threeDSAuthorisationRequired = $resp->getData( 'ThreeDSAuthorisationRequired' );
            if ( !is_null( $threeDSAuthorisationRequired ) ) {
                $aprilPaymentAction->reset();
                $aprilPaymentAction->setPaymentActionRequired( $threeDSAuthorisationRequired );
                $aprilPaymentAction->setAmount( $amount, $currencyCode );
                $aprilPaymentAction->setAprilOrderId( $aprilOrderId );
                $aprilPaymentAction->setPaymentToken( $aprilPaymentToken );
                $aprilPaymentAction->setRequest3DS( $request3DS );
                $aprilPaymentAction->save();

                PrestaShopLogger::addLog(
                    'Redirect URL: ' . $threeDSAuthorisationRequired['redirectURL'],
                    1,
                    AprilHelper::E_CODE_3DS_REDIRECTION_REQUIRED,
                    'AprilTransaction',
                    1
                );
                throw new AprilRedirectException( $threeDSAuthorisationRequired['redirectURL'] );
            }

            PrestaShopLogger::addLog(
                'Payment failed: April order pay API response is incorrect ',
                1,
                AprilHelper::E_CODE_APRIL_FAILED_PAYMENT,
                'AprilTransaction',
                1
            );
            throw new Exception( $message );
        }
        else {
            $message = AprilHelper::getApiErrorMessage( $resp );
            $message = empty( $message ) ? 'Payment failed' : $message;

            PrestaShopLogger::addLog(
                'Payment failed: ' . $message,
                1,
                AprilHelper::E_CODE_APRIL_FAILED_PAYMENT,
                'AprilTransaction',
                1
            );
            throw new Exception( $message );
        }
        

    }
}
