<?php
/**
 * Copyright © April Solutions.
 * This file is part of April payments module for PrestaShop.
 *
 * @author    April Solutions (https://www.meetapril.com/)
 * @copyright April Solutions
 * @license   https://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 */

class AprilPaymentModuleFrontController extends AprilAbstractModuleFrontController
{
    public function processPayment()
    {
        session_start();
        PrestaShopLogger::addLog(
            'April payments Token: ' . $_POST['april-payment-token'] . 'April payments Data: ' . $_POST['april-payment-data'],
            1,
            AprilHelper::E_CODE_PAYMENT_PROCESS,
            "AprilPaymentController",
            1
        );
        $aprilPaymentToken = $_POST['april-payment-token'];
        $aprilPaymentData = json_decode($_POST['april-payment-data']);  // v1.0.1 Overrides the payment data
        $orderCurrency = $this->getOrderCurrency();
        $orderTotal = $this->getOrderGrandTotal();

        $aprilPaymentData = $this->_fetchPaymentData($aprilPaymentToken);  // v1.0.1 Fetched payment token data from April API
        $aprilOrder = $this->_createAprilOrder();
        $aprilTransaction = $this->_createAprilTransaction( $aprilOrder->orderId, $orderTotal, $orderCurrency, $aprilPaymentToken, $aprilPaymentData  );

        $this->completeOrder( $aprilTransaction->transactionId, $aprilOrder->orderId );
    }

    protected function additionalValidations() {
        if( !isset( $_POST['april-payment-token'] ) ){
            $this->_paymentProcessingError( 'No April payments token found', 'index.php?controller=order&step=4');
            return false;
        };
        return true;
    }

    private function _createAprilOrder() {

        $aprilOrder = new AprilOrder();

        $aprilOrder->orderTotal = $this->getOrderGrandTotal();
        $aprilOrder->currencyCode = $this->getOrderCurrency()->iso_code;
        $aprilOrder->billing = $this->_getBilling();
        $aprilOrder->shipping = $this->_getShipping();
        $aprilOrder->products = $this->_getProducts();

        $aprilOrder->create();

        return $aprilOrder;
    }

    private function _fetchPaymentData( $paymentToken ) {

        $aprilPaymentToken = new AprilPaymentToken();
        $aprilPaymentToken->fetch($paymentToken);

        return $aprilPaymentToken->method;
    }

    private function _createAprilTransaction( $aprilOrderId, $amount, $currencyCode, $aprilPaymentToken, $aprilPaymentData  ) {

        $request3DS = $this->_getRequest3DS( $amount );
        $aprilPaymentAction = new AprilPaymentAction();

        $aprilTransaction = new AprilTransaction();
        $aprilTransaction->create( $aprilOrderId, $amount, $currencyCode, $aprilPaymentToken, $aprilPaymentData, $aprilPaymentAction, $request3DS );

        return $aprilTransaction;
    }

    private function _getBilling() {
        $billing = array();
        $address = array();
        $billing_address = new Address( $this->context->cart->id_address_invoice );
        $country_object = new Country( $billing_address->id_country );
        $state_object = new State( $billing_address->id_state );

        $address['line1'] = $billing_address->address1;
        $address['line2'] = $billing_address->address2;
        $address['city'] = $billing_address->city;
        if( !empty( $billing_address->id_state ) && !empty( $state_object ) ) {
            $address['state']   =   $state_object->iso_code;
        }
        $address['postalCode'] = $billing_address->postcode;
        $address['country'] = $country_object->iso_code;

        $billing['address'] = $address;
        $billing['name'] = $billing_address->firstname . ' ' . $billing_address->lastname;
        $billing['phoneNumber'] = $billing_address->phone;

        return $billing;
    }

    private function _getShipping()
    {
        $shipping = array();
        $address = array();
        $cart = $this->context->cart;

        if ( isset( $cart->id_address_delivery ) ) {
            $shipping_address = new Address( $cart->id_address_delivery );
            $carrier = new Carrier( $cart->id_carrier );
            $country_object = new Country( $shipping_address->id_country );
            $state_object = new State( $shipping_address->id_state );

            $address['line1'] = $shipping_address->address1;
            $address['line2'] = $shipping_address->address2;
            $address['city'] = $shipping_address->city;
            if( !empty( $shipping_address->id_state ) && !empty( $state_object ) ) {
                $address['state']   =   $state_object->iso_code;
            }
            $address['postalCode'] = $shipping_address->postcode;
            $address['country'] = $country_object->iso_code;

            $shipping['amount'] = AprilHelper::convertToCents( $cart->getPackageShippingCost() );
            $shipping['address'] = $address;
            $shipping['carrier'] = $carrier->name;
            $shipping['name'] = $shipping_address->firstname . ' ' . $shipping_address->lastname;
            $shipping['phoneNumber'] = $shipping_address->phone;
        }

        return $shipping;
    }

    private function _getProducts()
    {
        $products = array();
        $psProducts = $this->context->cart->getProducts( true );
        if ( count( $psProducts ) > 0) {
            foreach( $psProducts as $psProduct ) {

                $tmp = array(
                    'amount' => array(
                        'minorCurrencyUnits' => AprilHelper::convertToCents( $psProduct['price_wt'] ),
                        'currency' => strval( $this->getOrderCurrency()->iso_code )
                    ),
                    'description' => (string) $psProduct['description_short'],
                    'sku' => (string) $psProduct['reference'],
                    'quantity' => (int) $psProduct['quantity'],
                    'imageUrl' => $this->_getProductImageUrl( $psProduct['id_product'] )
                );
                array_push( $products, $tmp );
            }
        }
        return $products;
    }

    private function _getProductImageUrl( $productId )
    {
        $link = new Link();
        $product = new Product( (int) $productId, false, $this->context->cart->id_lang );
        $img = $product->getCover( $product->id );

        return $link->getImageLink( isset( $product->link_rewrite ) ? $product->link_rewrite : $product->name, (int) $img['id_image'] );
    }

    private function _getRequest3DS( $amount )
    {
        $request3DSSetting = (boolean) Configuration::get( 'APRIL_REQUEST_3DS' );
        $minAmt3DSSetting = (float) Configuration::get( 'APRIL_MINIMUM_AMOUNT_3DS' );
        return ( $request3DSSetting && $amount >= $minAmt3DSSetting );
    }
}
