'use strict';

/* API Includes */
var Resource = require('dw/web/Resource');

/* Script Includes */
var CommonUtils = require('*/cartridge/scripts/common/limepayCommonUtils');

const limepayPaymentMethodID = 'Limepay';

/**
 * Method to process refund for any limepay order
 * @param {Object} order - Order whose refund is to be processed
 * @param {Number} refundAmount - amount to be refunded for the order
 * @returns {Object} response - response data
 */
function ProcessRefund(order, amount) {
    var LimepayRefundService = require('*/cartridge/scripts/services/limepayRefundService');

    var refundAmount = parseFloat(amount);
    var response = null;
    var limepayPaymentInstrument = null;
    var orderID = order.orderNo;

    if (!order) {
        CommonUtils.logDebug('Order could not be found for order ID : ' + orderID);
        response = {
            ok: false,
            errorMessage: '{"message": "Order could not be found"}'
        };
    } else if (isNaN(refundAmount) || (!isNaN(refundAmount) && refundAmount <= 0)) {
        CommonUtils.logDebug('Invalid refund amount requested : ' + refundAmount);
        response = {
            ok: false,
            errorMessage: '{"message": "Invalid refund amount"}'
        };
    } else {
        // Retrieve limepay payment instrument for order
        var iter = order.getPaymentInstruments().iterator();
        while (iter.hasNext()) {
            var paymentInstrument = iter.next();
            if (paymentInstrument.paymentMethod === limepayPaymentMethodID) {
                limepayPaymentInstrument = paymentInstrument;
                break;
            }
        }

        if (limepayPaymentInstrument) {
            // Limepay refund API call
            // Executes only when order has limepay payment instrument
            response = LimepayRefundService.createRefund(order, refundAmount, limepayPaymentInstrument);
        } else {
            response = {
                ok: false,
                errorMessage: '{"message": "Could not find limepay payment instrument for order, unable to process refund."}'
            };
        }
    }

    if (!response.ok) {
        CommonUtils.logError(Resource.msg('limepay.refund.failed', 'limepay', null));
        sendRefundFailedNotification(response, orderID, limepayPaymentInstrument, refundAmount);
    }

    return response;
}

/**
 * Method to send email notification for refund failure to customer service with details
 * @param {Object} responseObj - Response object for service handler
 * @param {String} orderID - Request id of order for refund
 * @param {dw.order.PaymentInstrument} paymentInstrument - limepay payment instrument of order
 * @param {Number} refundAmount - Requesred amount for refund
 */
function sendRefundFailedNotification(responseObj, orderID, paymentInstrument, refundAmount) {
    var Site = require('dw/system/Site');
    var Mail = require('dw/net/Mail');
    var HashMap = require('dw/util/HashMap');
    var Template = require('dw/util/Template');

    var refundNotificationRecipient = Site.getCurrent().getCustomPreferenceValue('limepayCustomerServiceEmail');
    var responseError = CommonUtils.getResponseErrorMsg(responseObj.errorMessage);

    if (!empty(refundNotificationRecipient)) {
        CommonUtils.logDebug('Sending refund failed email notification to : ' + refundNotificationRecipient);
        var emailObj = {
            to: refundNotificationRecipient,
            subject: Resource.msg('limepay.refund.failure.title', 'limepay', null),
            from: Site.current.getCustomPreferenceValue('customerServiceEmail') || 'no-reply@testorganization.com',
            template: 'emailLimepayRefundFail',
            context: {
                orderNo: orderID,
                orderAmount: paymentInstrument ? paymentInstrument.getPaymentTransaction().amount : 'N/A',
                refundAmount: refundAmount,
                errorCode: responseError.hasOwnProperty('errorCode') ? responseError.errorCode : 'N/A',
                errorMessage: responseError.message
            }
        };

        var email = new Mail();
        email.addTo(emailObj.to);
        email.setSubject(emailObj.subject);
        email.setFrom(emailObj.from);

        var context = new HashMap();
        Object.keys(emailObj.context).forEach(function (key) {
            context.put(key, emailObj.context[key]);
        });

        var template = new Template(emailObj.template);
        var content = template.render(context).text;
        email.setContent(content, 'text/html', 'UTF-8');
        email.send();

        CommonUtils.logDebug('Successfully sent email notification');
    }
}

/*
 * Module exports
 */
exports.ProcessRefund = ProcessRefund;
