'use strict';

/* Script Includes */
var CommonUtils = require('*/cartridge/scripts/common/limepayCommonUtils');
var limepayUtilsHelpers = require( '*/cartridge/scripts/helpers/limepayUtilsHelpers.js' );

const limepayServiceID = 'limepay.http.service';

/**
 * Makes a call to Limepay to generate Limepay order id
 *
 * @returns {dw.svc.Service} - The service response
 */
function getLimepayServiceDefinition() {
    var LocalServiceRegistry = require('dw/svc/LocalServiceRegistry');
    var service = null;

    try {
        service = LocalServiceRegistry.createService(limepayServiceID, {
            /**
             * A callback function to configure HTTP request parameters before
             * a call is made to Limepay create order web service
             *
             * @param {dw.svc.Service} svc Service instance
             * @param {string} requestObject - Request object, containing the end point, query string params, payload etc.
             * @returns {string} - The body of HTTP request
             */
            createRequest: function (svc, args, requestObject) {
                svc.addHeader('Limepay-SecretKey', args.secretKey);
                svc.addHeader('Content-Type', 'application/json');

                svc.setURL(svc.getURL() + args.urlPath);

                svc.setRequestMethod('POST');
                var jsonReq = JSON.stringify(requestObject);

                return jsonReq;
            },
            parseResponse: function (svc, httpClient) {
                return httpClient;
            },
            filterLogMessage: function (msg) {
                return msg;
            },
            mockCall: function (svc) {
            },
            getRequestLogMessage : function(requestObj) {
                try {
                    var jsonString = JSON.stringify(requestObj);
                    return jsonString;
                } catch(e) {}

                return requestObj;
            },
            getResponseLogMessage : function(responseObj) {
                if (responseObj instanceof dw.net.HTTPClient) {
                    try {
                        return responseObj.text;
                    } catch(e) {}
                }

                return responseObj;
            }
        });
        CommonUtils.logDebug('Successfully retrieved service for : ' + limepayServiceID);
    } catch (e) {
        CommonUtils.logError('Could not find service for name : ' + limepayServiceID);
    }

    return service;
}

function callService(args, requestObject) {
    var Resource = require('dw/web/Resource');
    var response = {};
    var result;

    try {
        if (!requestObject) {
            CommonUtils.logError(Resource.msg('limepay.invalid.request', 'limepay', null));
            throw new Error('Required requestObject parameter missing or incorrect.');
        }

        CommonUtils.logDebug('Invoking Limepay service for ' + args.type);
        result = getLimepayServiceDefinition().call(args, requestObject);

        if (result == null || result.status === dw.svc.Result.SERVICE_UNAVAILABLE || (!result.ok && !result.object)) {
            // Service is unavailable
            if (result.error === 403) {
                // Handling 3DS response
                response.ok = true;
                response.threeDSRedirect = true;
                response.object = JSON.parse(result.errorMessage);

                return response;
            }

            response.ok = false;
            response.errorCode = 'SERVICE_ERROR';
            if (result.status === dw.svc.Result.SERVICE_UNAVAILABLE || !result || !('errorMessage' in result)) {
                response.errorMessage = '{"message": "The service is down and we\'re unable to place your order now.  Please try again later."}';
            } else {
                response.errorMessage = result.errorMessage;
            }
            CommonUtils.logError('SERVICE_ERROR :' + response.errorMessage);
        } else {
            response.ok = true;
            response.object = JSON.parse(result.object.text);
        }
    } catch (e) {
        response.ok = false;
        response.errorMessage = '{"message": e.message}';
        CommonUtils.logError('SERVICE_ERROR :' + e.message);
    }

    return response;
}

exports.call = callService;
