'use strict';

/* API Includes */
var Site = require('dw/system/Site');
var PaymentMgr = require('dw/order/PaymentMgr');
var BasketMgr = require('dw/order/BasketMgr');

const limepayMainPaymentMethodID = 'Limepay';
// Dummy payment method for pay later threshold lookup
const limepayBNPLPaymentMethodID = 'Limepay_instalment';

/**
 * Check if Limepay payment is enabled through site preferences &
 * payment method active status
 * @returns {Boolean} - limepay enable status
 */
function isLimepayPaymentEnabled() {
    var limepayPaymentMethod = PaymentMgr.getPaymentMethod(limepayMainPaymentMethodID);
    if (Site.getCurrent().getCustomPreferenceValue('limepayEnabled') && limepayPaymentMethod.isActive() && limepayPaymentMethod.isApplicable(customer, null, null)) {
        return true;
    }
    return false;
}

/**
 * Check if Limepay 3DS payment is enabled through site preferences
 * @returns {Boolean} - limepay 3DS enable status
 */
function isLimepay3DSEnabled() {
     return !!Site.getCurrent().getCustomPreferenceValue('3DSEnabled');
}

/**
 * Checks current customer groups to exclude pay later option
 * for the basket customer
 * @return {boolean} - disable pay later status for customer
 */
function isPayLaterExcludedForCustomer() {
    if (!isLimepayPaymentEnabled()) {
        return true;
    }

    var currentCustomerGroups = session.customer.customerGroups;
    var disablePayLaterOption = false;
    if (!empty(currentCustomerGroups)) {
        for (var i = 0; i < currentCustomerGroups.length; i++) {
            if (currentCustomerGroups[i].ID === 'ExcludeLimepayPayLater') {
                disablePayLaterOption = true;
                break;
            }
        }
    }
    return disablePayLaterOption;
}

/**
 * Get the Limepay context object to initialize the Limepay payment form
 * @return {Object} Limepay context global object
*/
function getWidgetContextObject() {
    var URLUtils = require('dw/web/URLUtils');
    return {
        isLimepayEnabled: isLimepayPaymentEnabled(),
        publicKey: Site.getCurrent().getCustomPreferenceValue('limepayClientPublicKey'),
        hidePayLaterOption: isPayLaterExcludedForCustomer(),
        hideFullPayOption: Site.getCurrent().getCustomPreferenceValue('limepayPaymentOptions') != 'multiple',
        preselectedPaymentType: getWidgetTogglePreselection(),
        preferredPaymentType: Site.getCurrent().getCustomPreferenceValue('limepayPaymentOptions') == 'multiple' ? 'multiple' : 'splitOnly',
        basketContextURL: URLUtils.url('Limepay-BasketContext').toString(),
        savePaymentTokenURL: URLUtils.url('Limepay-SavePaymentToken').toString(),
        saveToggleSelectionURL: URLUtils.url('Limepay-SaveSelection').toString(),
        customerLoginURL: URLUtils.url('Limepay-SigninCustomer').toString(),
        checkoutCSSOverride: URLUtils.httpsStatic('/css/limepay.css').toString(),
        bnplCheckoutThresholdExceeds: bnplContextThresholdExceeds('checkout')
    };
}

/**
 * Get the customer Basket Context Object
 * @return {Object} Customer basket
*/
function getBasketData() {
    var BasketMgr = require('dw/order/BasketMgr');
    var LimepayModel = require('~/cartridge/scripts/common/limepayModel');

    var currentBasket = BasketMgr.getCurrentBasket();
    return currentBasket ? new LimepayModel(currentBasket) : {};
}

/**
 * Checks whether the product or its primary category is elligible for
 * Limepay pay later option
 * @param {dw.catalog.Product} product - product whose elligibility is to be found
 * @return {boolean} product elligibility status
 */
function excludeProductPayLater(product) {
    if (!isLimepayPaymentEnabled()) {
        return true;
    }

    if (product.custom.disableLimepayPayLater) {
        return true;
    } else if (product.variant && product.masterProduct.custom.disableLimepayPayLater) {
        // Get disablity status for variant from master
        return true;
    } else if (product.productSet) {
        // Check if product set, products qualify for Pay later option
        var setProductExcluded = false;
        var productSetProducts = product.productSetProducts;
        for (var i = 0; i < productSetProducts.length; i++) {
            if (excludeProductPayLater(productSetProducts[i])) {
                setProductExcluded = true;
            }
        }
        return setProductExcluded;
    } else if (!empty(product.primaryCategory) && product.primaryCategory.custom.disableLimepayPayLater) {
        // If no exlusion found in product level, search in primary category
        return true;
    } else if (product.variant || product.variationGroup) {
        // If viewed product is variant or variation group, and have no primary category
        // look up from master product
        var masterProduct = product.masterProduct;
        return !empty(masterProduct.primaryCategory) && masterProduct.primaryCategory.custom.disableLimepayPayLater ? true : false;
    }
    return false;
}

/**
 * Get the limepay payment widget display mode based on context
 * @return {String} limepayPaymentWidgetMode - context widget display mode
*/
function getLimepayWidgetDisplayMode(context) {
    if (!isLimepayPaymentEnabled()) {
        return 'hidden';
    }

    var limepayPaymentOptions = Site.getCurrent().getCustomPreferenceValue('limepayPaymentOptions');

    // If payment configuration preference is 'splitOnly'
    // show only textual widget irrespective of product or cart configuration
    var limepayPaymentWidgetMode = 'textual';

    if (limepayPaymentOptions != 'splitOnly') {
        // Payment configuration preference is 'multiple'
        if (!empty(context) && context == 'product') {
            // Product context widget display mode
            limepayPaymentWidgetMode = Site.getCurrent().getCustomPreferenceValue('limepayPDPWidgetMode') == 'toggle' ? 'toggle' : 'textual';
        } else if (!empty(context) && context == 'cart') {
            // Cart context widget display mode
            limepayPaymentWidgetMode = Site.getCurrent().getCustomPreferenceValue('limepayCartWidgetMode') == 'toggle' ? 'toggle' : 'textual';
        }
    }

    return limepayPaymentWidgetMode;
}

/**
 * If Limepay pay later payment method threshold exceeds
 * disable pay later option from widget display or checkout based on context
 * @returns {Boolean} - bnpl option allowed
 */
function bnplContextThresholdExceeds(context) {
    if (!isLimepayPaymentEnabled()) {
        return true;
    }

    var paymentAmount;
    var currentBasket = BasketMgr.getCurrentBasket();
    if (currentBasket !== null && !empty(context)) {
        if (context == 'cart') {
            if (currentBasket.totalGrossPrice.available) {
                paymentAmount = currentBasket.totalGrossPrice;
            } else {
                paymentAmount = currentBasket.getAdjustedMerchandizeTotalPrice(true).add(currentBasket.giftCertificateTotalPrice);
            }
        } else if (context == 'checkout') {
            var limepayCheckoutHelpers = require('~/cartridge/scripts/helpers/limepayCheckoutHelpers');
            paymentAmount = limepayCheckoutHelpers.getNonGiftCertificateAmount(currentBasket);
        }

        var limepayBNPLPaymentMethod = PaymentMgr.getPaymentMethod(limepayBNPLPaymentMethodID);
        if (limepayBNPLPaymentMethod !== null && !limepayBNPLPaymentMethod.isApplicable(customer, null, paymentAmount.value)) {
            return true;
        }
    }

    return false;
}

/**
 * Retrieves the toggle preselection for the user, based on session versus site preferences
 * with first priority given to user session
 * @returns {String} - widget toggle preselection value
 */
function getWidgetTogglePreselection() {
    var toggleUserSelection = 'limepayToggleSelection' in session.privacy ? session.privacy.limepayToggleSelection : '';
    var preferredToggleSelection = Site.getCurrent().getCustomPreferenceValue('limepayMultiOptionDefault');

    return !empty(toggleUserSelection) ? toggleUserSelection : (preferredToggleSelection == 'full' ? 'full' : 'split');
}

/**
 * Check if Limepay payment method is allowed for customer at checkout
 * to place order, based on 'limepayPaymentOptions' preference
 * and current customer state. At the least Limepay one time payment option is provided for customer checkout
 * for limepayPaymentOptions preference being 'multiple' even if an item in cart disqualifies pay later
 * @returns limepayCheckoutAllowed - limepay payment method allowed at checkout
 */
function limepayCheckoutAllowed() {
    var limepayCheckoutAllowed = true;
    var limepayPaymentOptions = Site.getCurrent().getCustomPreferenceValue('limepayPaymentOptions');

    if (!isLimepayPaymentEnabled()) {
        // Limepay disabled for site
        limepayCheckoutAllowed = false;
    } else if (limepayPaymentOptions == 'splitOnly' && isPayLaterExcludedForCustomer()) {
        // splitOnly, if customer have disqualifying item in cart
        // limepay checkout disabled
        limepayCheckoutAllowed = false;
    }

    return limepayCheckoutAllowed;
}

/**
 * Check if we have Limepay in container
 * @returns isLimePay - to determin container has limepay or not
 */
function isPaymentMethodLimePay() {
    var isLimePay;
    var currentBasket = BasketMgr.getCurrentBasket();

    if (!currentBasket) {
        return isLimePay;
    }

    var LIME_PAY_METHODS = [limepayMainPaymentMethodID, limepayBNPLPaymentMethodID];
    var orderPaymentInstruments = currentBasket.getPaymentInstruments();

    isLimePay = orderPaymentInstruments.toArray().some(function(paymentInstrument) {
        var paymentMethodName = '';

        if (paymentInstrument.paymentMethod) {
            var paymentMethod = PaymentMgr.getPaymentMethod(paymentInstrument.paymentMethod);
            paymentMethodName = paymentMethod ? paymentMethod.ID : '';
        }

        return LIME_PAY_METHODS.indexOf(paymentMethodName) > -1;
    });

    return isLimePay;
}

/**
 * Check if Limepay pay total is more than threshold
 * for 3DS Payments
 * @param {dw.order.Order | dw.order.Basket} lineItemCtnr
 * @returns {Boolean} - limepay 3DS is eligible or not
 */
function isOrderTotalEligibleFor3DS(lineItemCtnr) {
    if (!lineItemCtnr) {
        return false;
    }

    var limePayMin3DSAmount = Site.getCurrent().getCustomPreferenceValue('limePayMin3DSAmount');

    if (!isLimepay3DSEnabled() || empty(limePayMin3DSAmount)) {
        return true;
    }

    var orderTotal = lineItemCtnr.getTotalNetPrice().value;

    return orderTotal >= limePayMin3DSAmount;
}

/**
 * Retrieve limepay primary color from site preferences
 * @returns {String} primaryColor - primary color
*/

function getPrimaryColor() {
    var primaryColor = Site.getCurrent().getCustomPreferenceValue('limepayPrimaryColor');
    return primaryColor;
}

module.exports = {
    isLimepayPaymentEnabled: isLimepayPaymentEnabled,
    isPayLaterExcludedForCustomer: isPayLaterExcludedForCustomer,
    getWidgetContextObject: getWidgetContextObject,
    getBasketData: getBasketData,
    excludeProductPayLater: excludeProductPayLater,
    getLimepayWidgetDisplayMode: getLimepayWidgetDisplayMode,
    bnplContextThresholdExceeds: bnplContextThresholdExceeds,
    getWidgetTogglePreselection: getWidgetTogglePreselection,
    limepayCheckoutAllowed: limepayCheckoutAllowed,
    isLimepay3DSEnabled: isLimepay3DSEnabled,
    isPaymentMethodLimePay: isPaymentMethodLimePay,
    isOrderTotalEligibleFor3DS: isOrderTotalEligibleFor3DS,
    getPrimaryColor: getPrimaryColor
};
