'use strict';
var Site = require('dw/system/Site');

/**
 * Makes a payment transcation with Limepay
 * @param {dw.order.Order} order - order whose payment transaction is executed
 * @param {dw.order.PaymentInstrument} paymentInstrument - payment instrument of the order
 * @returns {Object} - Limepay payment transaction service response
 */
exports.createPayment = function (order, paymentInstrument) {
    var ServiceBuilder = require('~/cartridge/scripts/common/limepayServiceBuilder');
    var APIContextLib = require('~/cartridge/scripts/common/limepayAPIContext');
    var LimepayService = require('~/cartridge/scripts/services/limepayService');
    var CommonUtils = require('*/cartridge/scripts/common/limepayCommonUtils');
    var limepayUtilsHelpers = require( '*/cartridge/scripts/helpers/limepayUtilsHelpers.js' );

    var Transaction = require('dw/system/Transaction');
    var Order = require('dw/order/Order');

    var is3DSEnable = limepayUtilsHelpers.isLimepay3DSEnabled();
    var clientSecretKey = ServiceBuilder.getClientSecretKey();
    var limepayAPIContext = new APIContextLib();
    var url = limepayAPIContext.getLimepayAPIUrls().get('payment');
    var limepayOrderId = order.custom.limepayOrderId;
    url = url.replace('{order_id}', limepayOrderId);

    var paymentTransaction = paymentInstrument.getPaymentTransaction();

    var callArgs = {
        secretKey: clientSecretKey,
        urlPath: url,
        type: 'Payment'
    };

    var params = {
        order : order
    };

    var reqParams = request.httpParameterMap;
    var threeDSCallback = reqParams.isParameterSubmitted('threeDSCallback') ? reqParams.threeDSCallback.booleanValue : false;

    if (is3DSEnable && limepayUtilsHelpers.isOrderTotalEligibleFor3DS(order)) {
        params.request3DS = is3DSEnable;
    }

    if (threeDSCallback) {
        var limePayResponse = JSON.parse(paymentTransaction.custom.limepayPaymentResponse);

        if (limePayResponse && limePayResponse.paymentActionRequired) {
            params.paymentActionRequired = limePayResponse.paymentActionRequired;
        }
    }

    var requestBody = ServiceBuilder.getPaymentRequestBody(params);
    var response = LimepayService.call(callArgs, requestBody);

    if (response.ok && response.object) {
        CommonUtils.logDebug('Limepay payment service successful');
        Transaction.wrap(function () {
            if (!response.threeDSRedirect) {
                // Save payment transaction ID and response data
                paymentTransaction.setTransactionID(response.object.transactionId);
                order.setPaymentStatus(Order.PAYMENT_STATUS_PAID);
            }
            paymentTransaction.custom.limepayPaymentResponse = JSON.stringify(response.object);
        });
    } else if (!response.ok) {
        Transaction.wrap(function () {
            // Save payment failure response data
            paymentTransaction.custom.limepayPaymentResponse = JSON.stringify(response);
        });
    }

    return response;
};
