'use strict';

/**
 * Initiates an order transaction refund for the given amount
 * @param {String} orderID - order id for the order being refunded
 * @param {Number} refundAmount - amount being refunded
 * @returns {Object} - Limepay order refund service response
 */
exports.createRefund = function (order, refundAmount, paymentInstrument) {
    var ServiceBuilder = require('~/cartridge/scripts/common/limepayServiceBuilder');
    var APIContextLib = require('~/cartridge/scripts/common/limepayAPIContext');
    var LimepayService = require('~/cartridge/scripts/services/limepayService');
    var CommonUtils = require('*/cartridge/scripts/common/limepayCommonUtils');
    var Transaction = require('dw/system/Transaction');

    var response = null;

    // Retrieve limepay transaction ID from limepay payment instrument
    var limepayTransactionID = paymentInstrument.getPaymentTransaction().getTransactionID();
    if (!limepayTransactionID) {
        response = {
            ok: false,
            errorMessage: 'Invalid Limepay transaction ID for refund'
        };
        return response;
    }

    var orderCurrencyCode = order.getCurrencyCode();

    var clientSecretKey = ServiceBuilder.getClientSecretKey();
    var limepayAPIContext = new APIContextLib();
    var url = limepayAPIContext.getLimepayAPIUrls().get('refund');

    var callArgs = {
        secretKey: clientSecretKey,
        urlPath: url,
        type: 'Refund'
    };

    var requestBody = ServiceBuilder.getRefundRequestBody(limepayTransactionID, orderCurrencyCode, refundAmount);
    response = LimepayService.call(callArgs, requestBody);

    if (response.ok) {
        CommonUtils.logDebug('Limepay refund successfully processed');
        var refundResponseDetails = {
            refundId: response.object.refundId,
            amount: refundAmount
        };
        refundResponseDetails = JSON.stringify(refundResponseDetails);
        var orderRefundResponses = order.custom.limepayRefundResponse.slice(0);
        var tempRefundResponseArr = [];
        orderRefundResponses.forEach(function (response) {
            tempRefundResponseArr.push(response);
        });
        tempRefundResponseArr.push(refundResponseDetails);

        // Save refund transaction ID response data
        Transaction.wrap(function () {
            order.custom.limepayRefundResponse = tempRefundResponseArr;
        });
    }

    return response;
};
