/* API Includes */
var Transaction = require('dw/system/Transaction');
var Resource = require('dw/web/Resource');

/* Script Modules */
var CommonUtils = require('*/cartridge/scripts/common/limepayCommonUtils');

const limepayPaymentMethodID = 'Limepay';

/**
 * Verifies the required information for billing form is provided
 * @param {Object} req - The request object
 * @param {Object} paymentForm - the payment form
 * @param {Object} viewFormData - object contains billing form data
 * @returns {Object} an object that has error information or payment information
 */
function processForm(req, paymentForm, viewFormData) {
    var viewData = viewFormData;

    viewData.paymentMethod = {
        value: paymentForm.paymentMethod.value,
        htmlName: paymentForm.paymentMethod.value
    };

    return {
        error: false,
        viewData: viewData
    };
}

/**
 * Creates a Limepay payment instrument for the basket
 */
function Handle(basket, paymentInformation) {
    const currentBasket = basket;
    var serverErrors = [];
    var CheckoutHelpers = require('*/cartridge/scripts/helpers/limepayCheckoutHelpers');
    var PaymentMgr = require('dw/order/PaymentMgr');

    try {
        Transaction.wrap(function () {
            const result = CheckoutHelpers.removePaymentInstruments(currentBasket);

            if (result.error) {
                CommonUtils.logError(Resource.msg('limepay.remove.payinstrument.failed', 'limepay', null));
                return { fieldErrors: [], serverErrors: Resource.msg('limepay.remove.payinstrument.failed', 'limepay', null), error: result.error };
            }

            var paymentInstrument = currentBasket.createPaymentInstrument(
                limepayPaymentMethodID,
                CheckoutHelpers.getNonGiftCertificateAmount(currentBasket)
            );

            var paymentTransaction = paymentInstrument.getPaymentTransaction();
            var paymentProcessor = PaymentMgr.getPaymentMethod(limepayPaymentMethodID).getPaymentProcessor();
            paymentTransaction.setPaymentProcessor(paymentProcessor);
        });
        CommonUtils.logDebug('Limepay payment instrument successfully created');
    } catch (e) {
        var errorMsg = Resource.msg('limepay.create.payinstrument.failed', 'limepay', null) + ' : ' + e.message;
        CommonUtils.logError(errorMsg);

        serverErrors.push(e.message);
        return { fieldErrors: [], serverErrors: serverErrors, error: true };
    }

    return { fieldErrors: [], serverErrors: serverErrors, error: false };
}

/**
 *  Calls Limepay create order API, and subsequently executes payment authorisation
 */
function Authorize(orderNumber, paymentInstrument, paymentProcessor) {
    var OrderMgr = require('dw/order/OrderMgr');
    var LimepayOrderService = require('*/cartridge/scripts/services/limepayOrderService');
    var LimepayPaymentService = require('*/cartridge/scripts/services/limepayPaymentService');

    const order = OrderMgr.getOrder(orderNumber);
    var serverErrors = [];
    var reqParams = request.httpParameterMap;
    var threeDSCallback = reqParams.isParameterSubmitted('threeDSCallback') ? reqParams.threeDSCallback.booleanValue : false;

    // Only create order in Limepay before 3DS intiatation, ignore it during 3DS verification
    if (!threeDSCallback) {
        // Limepay create order API call
        const orderResult = LimepayOrderService.createLimepayOrder(order, paymentInstrument);

        if (!orderResult.ok) {
            // Failed to create order at Limepay, authorisation call cannot be proceeded
            var errorMessage = CommonUtils.getResponseErrorMsg(orderResult.errorMessage);
            CommonUtils.logError(Resource.msg('limepay.createorder.failed', 'limepay', null));
            return { fieldErrors: [], serverErrors: errorMessage.detail, error: true };
        }
    }

    // Limepay create payment API call
    const paymentResult = LimepayPaymentService.createPayment(order, paymentInstrument);

    if (!paymentResult.ok) {
        // Payment failed at Limepay
        var errorMessage = CommonUtils.getResponseErrorMsg(paymentResult.errorMessage);
        CommonUtils.logError(Resource.msg('limepay.payment.failed', 'limepay', null));
        return { fieldErrors: [], serverErrors: errorMessage.detail, error: true };
    }

    CommonUtils.logDebug('Limepay payment authorisation completed successfully');

    return { fieldErrors: [], serverErrors: serverErrors, error: false , paymentResult: paymentResult || '' };
}
exports.processForm = processForm;
exports.Handle = Handle;
exports.Authorize = Authorize;
