'use strict';
/* API Includes */
var Site = require('dw/system/Site');

var LimepayServiceBuilder = {
    /**
	 * Method to generate the request body for Limepay create order service
	 * @param {dw.order.Order} order - order whose request to be generated
	 * @param {dw.order.PaymentInstrument} paymentInstrument - limepay paymentInstrument of orderS
	 * @returns {Object} data - create order service request data
	 */
    getCreateOrderRequestBody: function (order, paymentInstrument) {
        var data = {
            internalOrderId: order.orderNo,
            paymentAmount: this.getAmountObj(paymentInstrument),
            customerEmail: order.getCustomerEmail(),
            phoneNo: order.custom.limepayFormattedPhoneNo,
            description: 'SFCC Online Order', // Any merchant label
            items: this.getProductsObj(order.getAllProductLineItems()),
            discount: this.getDiscountsObj(order),
            billing: this.getAddressObj(order, 'billing'),
            shipping: this.getAddressObj(order, 'shipping')
        };
        return data;
    },

    /**
	 * Method to generate the request body for Limepay payment service
	 * @param {dw.order.Order} order - order whose request to be generated
	 * @returns {Object} data - payment service request data
	 */
    getPaymentRequestBody: function (params) {
        var data = {
            paymentToken: params.order.custom.limepayPaymentToken
        };

        if (params.request3DS) {
            data.request3DS = params.request3DS || false;
            data.returnURL = params.returnURL;
        }

        if (params.paymentActionRequired) {
            data.paymentActionRequired = params.paymentActionRequired
        }

        return data;
    },

    /**
	 * Method to generate the request body for Limepay refund service
	 * @param {String} transactionID - transaction ID referenced for refund
     * @param {Number} refundAmount - amount to be refunded
	 * @returns {Object} data - refund service request data
	 */
    getRefundRequestBody: function (transactionID, currencyCode, refundAmount) {
        var data = {
            transactionId: transactionID,
            refundAmount: this.getAmountObj(null, refundAmount, currencyCode)
        };
        return data;
    },

    /**
	 * Retrieve limepay service client secret key from site preferences
	 * @returns {String} secretKey - client secret key
	 */
    getClientSecretKey: function () {
        var secretKey = Site.getCurrent().getCustomPreferenceValue('limepayClientSecretKey');
        return secretKey;
    },

    /**
     * Method to prepare payment amount data
     * @param {dw.order.PaymentInstrument} paymentInstrument - limepay payment instrument of order
     * @returns {Object} paymentData - payment amount data
     */
    getAmountObj: function (paymentInstrument, amount, currencyCode) {
        var paymentData = {};
        if (paymentInstrument) {
            // Create order API flow
            paymentData = {
                amount: parseInt((paymentInstrument.paymentTransaction.amount * 100).toFixed(0)),
                currency: session.getCurrency().getCurrencyCode()
            };
        } else if (amount && currencyCode) {
            // Refund API flow
            paymentData = {
                currency: currencyCode,
                amount: (amount * 100).toFixed(0)
            };
        }
        return paymentData;
    },
    /**
	 * Method to calculate total discounts in order
	 * @param {dw.order.Order} order - order whose total discount is to be calculated
	 * @returns {Object} - total order discounts
	 */
    getDiscountsObj: function (order) {
        var discountAmt = 0;

        // Order level discounts
        var orderPriceAdjustments = order.getPriceAdjustments();
        for (var i = 0; i < orderPriceAdjustments.length; i++) {
            discountAmt += orderPriceAdjustments[i].price;
        }

        // Shipping level discounts
        var orderShipments = order.getShipments();
        for (var i = 0; i < orderShipments.length; i++) {
            var shippingPriceAdjustments = orderShipments[i].shippingPriceAdjustments;
            for (var j = 0; j < shippingPriceAdjustments.length; j++) {
                discountAmt += shippingPriceAdjustments[j].price;
            }
        }

        // Product level discounts
        var productLineItems = order.getProductLineItems();
        for (var i = 0; i < productLineItems.length; i++) {
            var priceAdjustments = productLineItems[i].priceAdjustments;
            for (var j = 0; j < priceAdjustments.length; j++) {
                discountAmt += priceAdjustments[j].price;
            }
        }

        // Convert discounts to postive integer
        discountAmt = parseInt((discountAmt * -100).toFixed(0));

        return {
            amount: discountAmt
        };
    },
    /**
	 * Method to generate address request object for Limepay service
	 * @param {dw.order.Order} order - order to retrieve billing or shipping details
     * @param {String} addressType - label whose address object is prepared
	 * @returns {Object} addressObj - result
	 */
    getAddressObj: function (order, addressType) {
        var addressObj = {};

        var orderAddress;
        if (addressType == 'billing') {
            orderAddress = order.getBillingAddress();
            addressObj['phone'] = order.custom.limepayFormattedPhoneNo;
        }
        else if (addressType == 'shipping') {
            orderAddress = order.getDefaultShipment().getShippingAddress();
            var amount = order.getShippingTotalGrossPrice();
            addressObj['amount'] = parseInt((amount * 100).toFixed(0));
        }

        addressObj['name'] = (orderAddress.fullName) ? orderAddress.fullName : '';

        var address = {};
        address.line1 = (orderAddress.getAddress1()) ? orderAddress.getAddress1() : '';
        address.line2 = (orderAddress.getAddress2()) ? orderAddress.getAddress2() : '';
        address.city = (orderAddress.getCity()) ? orderAddress.getCity() : '';
        address.state = (orderAddress.getStateCode()) ? orderAddress.getStateCode() : '';
        address.postalCode = (orderAddress.getPostalCode()) ? orderAddress.getPostalCode() : '';
        address.country = (orderAddress.getCountryCode() && orderAddress.getCountryCode().value) ? orderAddress.getCountryCode().value : '';

        addressObj.address = address;

        return addressObj;
    },

    /**
	 * Method to generate the products request object for Limepay service
	 * @param {dw.util.Collection} lineItems - all product line items in order
	 * @returns {Object} productsObj - result
	 */
    getProductsObj: function (lineItems) {
        var productsObj = [];
        for (var i = 0; i < lineItems.length; i++) {
            var lineItem = lineItems[i];
            var product = lineItem.getProduct();
            var prdObj = {};
            var qty = lineItem.getQuantity().value;
            prdObj.description = lineItem.getProductName();
            prdObj.sku = lineItem.getProductID();
            prdObj.imageUrl = product.getImage('large', 0).getURL().https().toString(); // TO DO
            prdObj.quantity = qty;
            // Require tweaks here for proper rounded figure
            prdObj.amount = parseInt(((lineItem.getGrossPrice() / qty) * 100).toFixed(0));
            prdObj.currency = session.getCurrency().getCurrencyCode();
            productsObj.push(prdObj);
        }
        return productsObj;
    },

    /**
	 * Method to generate the request body for Limepay upsert customer service
	 * @param {dw.customer.CustomerMgr} customer - order whose request to be generated
	 * @returns {Object} data - create order service request data
	 */
    getUpsertCustomerRequestBody: function (customer) {
        var data = {
            internalCustomerId: customer.profile.customerNo,
            emailAddress: customer.profile.email,
            givenName: customer.profile.firstName,
            familyName: customer.profile.lastName
        };
        return data;
    }
};

module.exports = LimepayServiceBuilder;
