'use strict';

// E.164 phone number format library
var libphonenumber = require('google-libphonenumber');

// Limepay global variables
var limepayPhone;
var formattedPhoneNo;
var limepayEmail;
var LimepayPayment = {
    currentStage: null,
    LimepayCheckoutContext: null,
    LimepayOrderData: null,
    LimepayPaymentPlanTermsAccepted: false,
    LimepayPaymentPlan: null
};

LimepayPayment.init = function (setup) {
    this.LimepayCheckoutContext = setup.config;
    this.LimepayOrderData = setup.order;

    this.currentStage = $('.data-checkout-stage').attr('data-checkout-stage');

    setInterval(function () {
        var currentCheckoutStage = $('.data-checkout-stage').attr('data-checkout-stage');
        if (currentCheckoutStage !== 'payment' && $('.limepay-payment-button').length > 0) {
            $('.limepay-payment-button').remove();
        } else if (currentCheckoutStage === 'payment' && this.currentStage !== 'payment' && isLimepayCheckoutTabActive()) {
            handleLimepayCheckoutTab(this.LimepayCheckoutContext, this.LimepayOrderData);
        }
        this.currentStage = currentCheckoutStage;
    }.bind(this), 100);
};

/**
 * Checks if Limepay payment tab is active
 * @returns {Boolean} - limepay payment tab active status
 */
function isLimepayCheckoutTabActive() {
    return $('.payment-options .nav-item .active').parent().attr('data-method-id') === 'Limepay';
}

/**
 * Save the generated Limepay payment token to basket, for completing order
 * @param {String} token - Limepay generated token
 */
function savePaymentToken(token) {
    if (window.LimepayCheckout && window.LimepayCheckout !== undefined && token !== undefined) {
        var saveTokenURL = window.LimepayContext.savePaymentTokenURL;
        var csrfTokenName = $('#limepay-checkout').attr('data-token-name');
        var csrfTokenValue = $('#limepay-checkout').attr('data-token-value');
        saveTokenURL = saveTokenURL + '?' + csrfTokenName + '=' + csrfTokenValue;
        $.spinner().start();
        var data = {
            paymentToken: token,
            phone: formattedPhoneNo
        };
        $.ajax({
            type: 'post',
            async: false,
            dataType: 'json',
            data: data,
            url: saveTokenURL
        })
            .done(function (data) {
                if (data && data.error && data.redirectUrl) {
                    location.href = data.redirectUrl;
                } else {
                    $('body').trigger('limePayment:postSubmitLimePayTokenEvents');
                }
            })
            .fail(function (data) {
                if (data.responseJSON.redirectUrl) {
                    window.location.href = data.responseJSON.redirectUrl;
                }
            })
            .always(function () {
                $.spinner().stop();
            });
    }
}

/**
 * Initializes or re-renders Limepay checkout form
 * @param {Object} context - Limepay checkout form configuration
 * @param {Object} order - customer order data for Limepay checkout
 */
function initLimepayCheckout(context, order) {
    // change any other submit button with this extension before loading the widget
    $('body').trigger('limePayment:handleLimePaySubmitButton');

    limepayEmail = getCustomerEmail(order);

    formattedPhoneNo = getFormattedPhoneNo();

    var billingDetails = getBillingDetails();
    var limepayPrimaryColor = getPrimaryColor();
    var limepayCustomerToken = getCustomerToken();

    LimepayCheckout.init({
        publicKey: context.publicKey,
        email: limepayEmail,
        phone: formattedPhoneNo,
        customerFirstName: billingDetails.firstName,
        customerLastName: billingDetails.lastName,
        customerResidentialAddress: billingDetails.customerResidentialAddress,
        customerToken: limepayCustomerToken,
        hidePayLaterOption: context.hidePayLaterOption,
        hideFullPayOption: context.hideFullPayOption,
        paymentType: context.preselectedPaymentType == 'full' ? 'paycard' : 'payplan',
        paymentToken: function (token) {
            savePaymentToken(token);
        }
    });
    LimepayCheckout.render({
        elementId: 'limepay-cont',
        currency: order.orderInfo.currency,
        amount: order.orderInfo.amount,
        paymentType: context.preselectedPaymentType == 'full' ? 'paycard' : 'payplan',
        showPayNow: false,
        showPayPlanSubmit: false,
        checkoutCSSOverride: context.checkoutCSSOverride,
        primaryColor: limepayPrimaryColor
    });
    LimepayCheckout.errorHandler(function (err) {
        // Handle errors
    });
    LimepayCheckout.eventHandler(function (event) {
        $('body').trigger('limePayment:handleLimePayEvents', { event: event });
    });
}

function handleLimepayCheckoutTab(config, order) {
    if (window.LimepayContext.isLimepayEnabled) {
        initLimepayCheckout(config, order);
    }
}

function getCustomerEmail(orderContext) {
    // Registered customer email from context
    var email = orderContext.customerEmail;

    // Guest email field
    if ($('input[name$=_coCustomer_email]').length > 0) {
        email = $('input[name$=_coCustomer_email]').val() ? $('input[name$=_coCustomer_email]').val() : $('.customer-summary-email').text();
    }
    return email;
}

/**
 * Parses billing phone number & country to generate phone in E164 format
 * @returns {String} e164FormattedPhoneNo - phone number in E164 format
 */
function getFormattedPhoneNo() {
    // Billing form values
    limepayPhone = $('input[name$=_billing_contactInfoFields_phone]').val();
    var billingCountryCode = $('select[name$=_billing_addressFields_country]').val();
    var e164FormattedPhoneNo = '';

    if (limepayPhone && billingCountryCode) {
        var PNF = libphonenumber.PhoneNumberFormat;
        var phoneUtil = libphonenumber.PhoneNumberUtil.getInstance();
        var number = phoneUtil.parseAndKeepRawInput(limepayPhone, billingCountryCode);
        e164FormattedPhoneNo = phoneUtil.format(number, PNF.E164);
    }

    return e164FormattedPhoneNo;
}

/**
 * Retrieve billing details for Limepay checkout rendering
 * @returns {Object} - customer billing details
 */
function getBillingDetails() {
    // Billing customer
    var firstName = $('input[name$=_billing_addressFields_firstName]').val();
    var lastName = $('input[name$=_billing_addressFields_lastName]').val();

    // Billing Address
    var address1 = $('input[name$=_billing_addressFields_address1]').val();
    var address2 = $('input[name$=_billing_addressFields_address2]').val();
    var city = $('input[name$=_billing_addressFields_city]').val();
    var country = $('select[name$=_billing_addressFields_country] option:selected').text().trim();
    var stateCode = $('select[name$=_billing_addressFields_states_stateCode]').val();
    var postalCode = $('input[name$=_billing_addressFields_postalCode]').val();

    var residentialAddress = '';
    if (address1) residentialAddress += address1 + ', ';
    if (address2) residentialAddress += address2 + ', ';
    if (city) residentialAddress += city + ' ';
    if (stateCode) residentialAddress += stateCode + ' ';
    if (postalCode) residentialAddress += postalCode + ', ';
    if (country) residentialAddress += country;

    return {
        firstName: firstName,
        lastName: lastName,
        customerResidentialAddress: residentialAddress
    };
}

/**
 * Parses limepay primary color
 * @returns {String} limepayPrimaryColor - primary color
 */
function getPrimaryColor() {
    var limepayPrimaryColor = $('input[name$=limepay-primary-color]').val();
    return limepayPrimaryColor;
}

/**
 * Parses limepay customer token
 * @returns {String} limepayCustomerToken - customer token
 */
function getCustomerToken() {
    var limepayCustomerToken = $('input[name$=limepay-customer-token]').val();
    return limepayCustomerToken;
}

module.exports = function () {
    window.LimepayContext = $('#limepay-checkout-context').data('checkoutcontext');
    window.LimepayOrderData = $('#limepay-order-data').data('orderdata');

    LimepayPayment.init({
        config: window.LimepayContext,
        order: window.LimepayOrderData
    });

    $('body').on('limePayment:handleLimePayEvents', function (e, data) {
        if (data.event && 'limepay_toggled_payment_type' == data.event.eventName) {
            LimepayPayment.LimepayPaymentPlan = data.event.eventData.paymentType;
            if ('paycard' === data.event.eventData.paymentType) {
                LimepayPayment.LimepayPaymentPlanTermsAccepted = false;
            }

            // Save toggle selection change to user session
            var userToggleSelection = data.event.eventData.paymentType == 'payplan' ? 'split' : 'full';
            var saveSelectionURL = LimepayPayment.LimepayCheckoutContext.saveToggleSelectionURL;
            saveSelectionURL = saveSelectionURL + '?limepayToggle=' + userToggleSelection;

            // Update context object selection
            LimepayPayment.LimepayCheckoutContext.preselectedPaymentType = userToggleSelection;

            $.ajax({
                url: saveSelectionURL,
                success: function (data) {
                }
            });
        }

        if (data.event && 'limepay_first_payment_amount_changed' == data.event.eventName && !LimepayPayment.LimepayPaymentPlan) {
            LimepayPayment.LimepayPaymentPlan = 'payplan';
            LimepayPayment.LimepayPaymentPlanTermsAccepted = false;
        }

        if (data.event && 'limepay_payment_plan_terms_accepted' == data.event.eventName) {
            LimepayPayment.LimepayPaymentPlan = 'payplan';
            LimepayPayment.LimepayPaymentPlanTermsAccepted = true;
        }

        // Update Payment Button Based on Payment Plans
        $('body').trigger('limePayment:handlePaymentSubmitBtn');
    });

    $('body').on('limePayment:handlePaymentTabEvents', function () {
        handleLimepayCheckoutTab(LimepayPayment.LimepayCheckoutContext, LimepayPayment.LimepayOrderData);
    });

    $('body').on('limePayment:handlePaymentSubmitBtn', function () {
        if (LimepayPayment.LimepayPaymentPlan && 'paycard' == LimepayPayment.LimepayPaymentPlan) {
            $('.limepay-submit-payment').prop('disabled', false);
        } else if (LimepayPayment.LimepayPaymentPlan && 'payplan' == LimepayPayment.LimepayPaymentPlan) {
            $('.limepay-submit-payment').prop('disabled', !LimepayPayment.LimepayPaymentPlanTermsAccepted);
        }
    });

    $('body').on('limePayment:handleLimePaySubmitButton', function () {
        var submitPaymentBtn = $('.submit-payment');

        // Hide regular submit-payment button
        // and create dummy for limepay checkout
        submitPaymentBtn.addClass('d-none');
        var limepayBtnContainer = $('.limepay-payment-button');
        if ($(limepayBtnContainer).length == 0) {
            limepayBtnContainer = $('<div/>', { 'class': 'col-12 limepay-payment-button' });
            limepayBtnContainer.insertAfter($('.next-step-button'));

            var limepayPaymentBtn = $('.limepay-submit-payment');
            if ($(limepayPaymentBtn).length == 0) {
                // Clone of regular sumbit-payment button
                limepayPaymentBtn = submitPaymentBtn.clone();
            }

            limepayPaymentBtn.removeClass('submit-payment d-none').addClass('limepay-submit-payment');
            var checkoutContext = LimepayPayment.LimepayCheckoutContext;
            if ((checkoutContext.preferredPaymentType == 'splitOnly') ||
                (checkoutContext.preselectedPaymentType == 'split' && !checkoutContext.hidePayLaterOption && !checkoutContext.bnplCheckoutThresholdExceeds)) {
                limepayPaymentBtn.prop('disabled', true);
            }
            limepayBtnContainer.html(limepayPaymentBtn);
        }
    });

    $('body').on('limePayment:resetLimePayPaymentTab', function () {
        // Remove clone button, when non limepay payment method selected
        $('.limepay-payment-button').remove();

        $('.submit-payment').removeClass('d-none');
        LimepayPayment.LimepayPaymentPlanTermsAccepted = false;
    });

    $('body').on('limePayment:postSubmitLimePayTokenEvents', function () {
        // Trigger the regular place order button to proceed checkout
        $('.submit-payment').trigger('click');
    });

    $('body').on('click', '.limepay-submit-payment', function (e) {
        e.preventDefault();
        if (isLimepayCheckoutTabActive() && null !== LimepayCheckout) {
            LimepayCheckout.submit();
        }
    });

    $('body').on('limePayment:handleEditBillingSummary', function () {
        if (window.LimepayContext.isLimepayEnabled && isLimepayCheckoutTabActive()) {
            LimepayPayment.LimepayPaymentPlanTermsAccepted = false;
            handleLimepayCheckoutTab(LimepayPayment.LimepayCheckoutContext, LimepayPayment.LimepayOrderData);
        }
    });

    $('body').on('limePayment:postUpdateTotals', function (e, data) {
        if (LimepayPayment.LimepayCheckoutContext.isLimepayEnabled) {
            $.ajax({
                type: 'GET',
                async: false,
                data: data,
                dataType: 'json',
                url: LimepayPayment.LimepayCheckoutContext.basketContextURL
            })
                .done(function (data) {
                    if (typeof data === 'object' && data.error) {
                        location.href = data.redirectUrl;
                        return;
                    }
                    window.LimepayOrderData = LimepayPayment.LimepayOrderData = data.limepayOrderData;
                    window.LimepayContext = LimepayPayment.LimepayCheckoutContext = data.limepayContext;
                });
        }
    });

    $('body').on('limePayment:getCustomerLoginToken', function (e, data) {
        if (window.LimepayContext.isLimepayEnabled) {
            if (data.customer.registeredUser) {
                var registeredUserData = {
                    email: data.customer.profile.email
                };
                var customerLoginURL = window.LimepayContext.customerLoginURL;
                var csrfTokenName = $('#limepay-checkout').attr('data-token-name');
                var csrfTokenValue = $('#limepay-checkout').attr('data-token-value');
                var signinURL = customerLoginURL + '?' + csrfTokenName + '=' + csrfTokenValue;
                $.ajax({
                    type: 'post',
                    async: false,
                    dataType: 'json',
                    url: signinURL,
                    data: registeredUserData,
                    success: function (res) {
                        if (res.success) {
                            $('input[name$=limepay-customer-token]').val(res.customerToken);
                        } else {
                            $('input[name$=limepay-customer-token]').val('');
                        }
                    },
                    error: function (err) {
                        $('input[name$=limepay-customer-token]').val('');
                    }
                });
            }
        }
    });

    // Update to email or phone field, re-renders limepay checkout form
    var limepayDynamicFields = [
        'input[name$=_coCustomer_email]',
        'input[name$=_billing_contactInfoFields_phone]'
    ];
    limepayDynamicFields.forEach(function (selector) {
        $(selector).on('blur', function (e) {
            var entryUpdated = $(this).attr('name').indexOf('email') > -1 ? ($(this).val() != limepayEmail) : ($(this).val() != limepayPhone);
            if (isLimepayCheckoutTabActive() && entryUpdated) {
                initLimepayCheckout(LimepayPayment.LimepayCheckoutContext, LimepayPayment.LimepayOrderData);
            }
        });
    });
};
