'use strict';

var server = require('server');

var csrfProtection = require('*/cartridge/scripts/middleware/csrf');

var URLUtils = require('dw/web/URLUtils');

/**
 * Prepares the basket context data used for rendering
 * Limepay checout form
 */
server.get(
    'BasketContext',
    server.middleware.https,
    function (req, res, next) {
        var BasketMgr = require('dw/order/BasketMgr');
        var LimepayModel = require('*/cartridge/scripts/common/limepayModel');
        var limepayUtilsHelpers = require('*/cartridge/scripts/helpers/limepayUtilsHelpers');
        var currentBasket = BasketMgr.getCurrentBasket();
        if (!currentBasket) {
            res.json({
                redirectUrl: URLUtils.url('Cart-Show').toString(),
                error: true
            });
            return next();
        }
        var limepayContext = limepayUtilsHelpers.getWidgetContextObject();
        var limepayModel = new LimepayModel(currentBasket);
        res.json({
            success: true,
            limepayContext: limepayContext,
            limepayOrderData: limepayModel
        });
        next();
    }
);

/**
 * Save Limepay payment data : token and formattedPhoneNo
 * onto basket and used for Limepay create order service
 */
server.post(
    'SavePaymentToken',
    server.middleware.https,
    csrfProtection.validateAjaxRequest,
    function (req, res, next) {
        var BasketMgr = require('dw/order/BasketMgr');
        var URLUtils = require('dw/web/URLUtils');
        var Transaction = require('dw/system/Transaction');
        var CommonUtils = require('*/cartridge/scripts/common/limepayCommonUtils');
        var currentBasket = BasketMgr.getCurrentBasket();
        if (!currentBasket) {
            res.json({
                redirectUrl: URLUtils.url('Cart-Show').toString(),
                error: true
            });
            return next();
        }
        var token = req.form.paymentToken;
        var phone = req.form.phone;
        try {
            Transaction.wrap(function () {
                // Save token and formatted phone number to basket
                // used for create order service call
                if (!empty(token)) {
                    currentBasket.custom.limepayPaymentToken = token;
                } else {
                    throw new Error('Limepay token is missing');
                }
                if (!empty(phone)) {
                    currentBasket.custom.limepayFormattedPhoneNo = phone;
                } else {
                    throw new Error('Limepay phoneNo is missing');
                }
            });
        } catch (error) {
            CommonUtils.logError(error);
            res.json({
                error: true
            });
            return next();
        }
        res.json({
            success: true
        });
        next();
    }
);

/**
 * Remember Limepay toggle widget change in default selection
 * onto user session. Toggle widget to stay preselected with this state across site
 */
server.get(
    'SaveSelection',
    server.middleware.https,
    function (req, res, next) {
        var userSelection = request.httpParameterMap.limepayToggle.stringValue;
        session.privacy.limepayToggleSelection = userSelection;
        res.json({
            success: true,
            userSelection: userSelection
        });
        next();
    }
);

/**
 * Remote include to render dynamic Limepay user toggle widget
 */
server.get(
    'UserToggleWidget',
    server.middleware.https,
    function (req, res, next) {
        var params = request.httpParameterMap;
        var limepayUtilsHelpers = require('*/cartridge/scripts/helpers/limepayUtilsHelpers');
        var togglePreselection = limepayUtilsHelpers.getWidgetTogglePreselection();

        res.render('/limepay/limepayUserToggleWidget', {
            togglePreselection: togglePreselection,
            amount: parseFloat(params.amount),
            currency: session.currency.currencyCode
        });

        next();
    }
);

/**
 * Retrieve dynamic user toggle state for Limepay toggle widget
 * used in cached quick view`s contents
 */
server.get(
    'UserToggleState',
    server.middleware.https,
    function (req, res, next) {
        var limepayUtilsHelpers = require('*/cartridge/scripts/helpers/limepayUtilsHelpers');
        var togglePreselection = limepayUtilsHelpers.getWidgetTogglePreselection();

        res.json({
            success: true,
            togglePreselection: togglePreselection
        });
        next();
    }
);

server.post(
    'Limepay3DS',
    server.middleware.https,
    function (req, res, next) {
        var hooksHelper = require('*/cartridge/scripts/helpers/hooks');
        var COHelpers = require('*/cartridge/scripts/checkout/checkoutHelpers');
        var addressHelpers = require('*/cartridge/scripts/helpers/addressHelpers');

        var Resource = require('dw/web/Resource');
        var OrderMgr = require('dw/order/OrderMgr');
        var Order = require('dw/order/Order');
        var Transaction = require('dw/system/Transaction');

        var orderNo = req.querystring.OrderNo;
        var order = OrderMgr.getOrder(orderNo);

        var orderPaymentStatus = order.getPaymentStatus();

        if (orderPaymentStatus && orderPaymentStatus.value === Order.PAYMENT_STATUS_PAID) {
            res.json({
                error: true,
                errorMessage: Resource.msg('error.technical', 'checkout', null)
            });

            return next();
        }

        var handlePaymentResult = COHelpers.handlePayments(order, orderNo);
        var paymentResult = handlePaymentResult.paymentResult || {};

        if (handlePaymentResult.error) {
            var errorMessage = Resource.msg('error.payment.not.valid', 'checkout', null);

            if (session.privacy.limepayErrorMsg) {
                errorMessage = session.privacy.limepayErrorMsg;
                delete session.privacy.limepayErrorMsg;
            }

            res.json({
                error: true,
                errorStage: {
                    stage: 'payment',
                    step: 'paymentInstrument'
                },
                errorMessage: errorMessage
            });
            return next();
        }

        // if 3DS Redirect
        if (paymentResult.threeDSRedirect) {
            res.json({
                paymentResult: paymentResult.object,
                threeDSURL: URLUtils.url('Limepay-Limepay3DS', 'threeDSCallback', true, 'OrderNo', order.orderNo).toString()
            });

            return next();
        }

        // Handle custom processing post authorization
        var options = {
            req: req,
            res: res
        };
        var postAuthCustomizations = hooksHelper('app.post.auth', 'postAuthorization', handlePaymentResult, order, options, require('*/cartridge/scripts/hooks/postAuthorizationHandling').postAuthorization);
        if (postAuthCustomizations && Object.prototype.hasOwnProperty.call(postAuthCustomizations, 'error')) {
            res.json(postAuthCustomizations);
            return next();
        }

        if (handlePaymentResult.error) {
            res.json({
                error: true,
                errorMessage: Resource.msg('error.technical', 'checkout', null)
            });
            return next();
        }

        var fraudDetectionStatus = hooksHelper('app.fraud.detection', 'fraudDetection', order, require('*/cartridge/scripts/hooks/fraudDetection').fraudDetection);
        if (fraudDetectionStatus.status === 'fail') {
            Transaction.wrap(function () { OrderMgr.failOrder(order, true); });

            // fraud detection failed
            req.session.privacyCache.set('fraudDetectionStatus', true);

            res.json({
                error: true,
                cartError: true,
                redirectUrl: URLUtils.url('Error-ErrorCode', 'err', fraudDetectionStatus.errorCode).toString(),
                errorMessage: Resource.msg('error.technical', 'checkout', null)
            });

            return next();
        }

        // Places the order
        var placeOrderResult = COHelpers.placeOrder(order, fraudDetectionStatus);
        if (placeOrderResult.error) {
            res.json({
                error: true,
                errorMessage: Resource.msg('error.technical', 'checkout', null)
            });
            return next();
        }

        if (req.currentCustomer.addressBook) {
            // save all used shipping addresses to address book of the logged in customer
            var allAddresses = addressHelpers.gatherShippingAddresses(order);
            allAddresses.forEach(function (address) {
                if (!addressHelpers.checkIfAddressStored(address, req.currentCustomer.addressBook.addresses)) {
                    addressHelpers.saveAddress(address, req.currentCustomer, addressHelpers.generateAddressName(address));
                }
            });
        }

        if (order.getCustomerEmail()) {
            COHelpers.sendConfirmationEmail(order, req.locale.id);
        }

        // Reset usingMultiShip after successful Order placement
        req.session.privacyCache.set('usingMultiShipping', false);

        // TODO: Exposing a direct route to an Order, without at least encoding the orderID
        //  is a serious PII violation.  It enables looking up every customers orders, one at a
        //  time.
        res.json({
            error: false,
            orderID: order.orderNo,
            orderToken: order.orderToken,
            continueUrl: URLUtils.url('Order-Confirm').toString()
        });

        res.json({
            error: false
        });

        return next();
    }
);

/**
 * singin limepay customer with customer id
 * give response with limepay customer token object
 */
server.post(
    'SigninCustomer',
    server.middleware.https,
    csrfProtection.validateAjaxRequest,
    function (req, res, next) {
        var CustomerMgr = require('dw/customer/CustomerMgr');
        var limepayUtilsHelpers = require('*/cartridge/scripts/helpers/limepayUtilsHelpers');
        var limepaySignInCustomerWithCustomerId = require('*/cartridge/scripts/services/limepaySignInCustomerWithCustomerId');
        var limepayUpsertCustomerService = require('*/cartridge/scripts/services/limepayUpsertCustomerService');
        var email = req.form.email;
        var limepayEnabled = limepayUtilsHelpers.isLimepayPaymentEnabled();
        var success = false;
        var customerToken = '';
        if (limepayEnabled) {
            var customer = CustomerMgr.getCustomerByLogin(email);
            var limepayCustomerId = customer.profile.custom.limepayCustomerId;
            if (!limepayCustomerId) {
                // Limepay create upsert customer API call
                var response = limepayUpsertCustomerService.upsertLimepayCustomer(customer);
                limepayCustomerId = response.object.customerId;
            }
            if (limepayCustomerId) {
                // Limepay customer login with customer id API call
                var response = limepaySignInCustomerWithCustomerId.signInLimepayCustomerWithCustomerId(limepayCustomerId);
                if (response.ok && response.object) {
                    success = true;
                    customerToken = response.object.customToken;
                }
            }
        }

        res.json({
            success: success,
            customerToken: customerToken
        });
        next();
    }
);

module.exports = server.exports();
