'use strict';

/* API Includes */
var URLUtils = require('dw/web/URLUtils');
var Transaction = require('dw/system/Transaction');

/* Script Modules */
var app = require('*/cartridge/scripts/app');
var guard = require('*/cartridge/scripts/guard');

var Order = app.getModel('Order');

/**
 * Prepares the basket context data used for rendering
 * Limepay checout form
 */
function basketContext() {
    var BasketMgr = require('dw/order/BasketMgr');
    var LimepayModel = require('*/cartridge/scripts/common/limepayModel');
    var currentBasket = BasketMgr.getCurrentBasket();

    var responseJson = {};

    if (!currentBasket) {
        responseJson.redirectUrl = URLUtils.url('Cart-Show').toString();
        responseJson.error = true;
        app.getView({
            JSONResponse: responseJson
        }).render('util/responsejson');
        return;
    }
    responseJson = new LimepayModel(currentBasket);
    app.getView({
        JSONResponse: responseJson
    }).render('util/responsejson');
}

/**
 * Save Limepay payment data : token and formattedPhoneNo
 * onto basket and used for Limepay create order service
 */
function savePaymentToken() {
    var BasketMgr = require('dw/order/BasketMgr');
    var CustomerMgr = require('dw/customer/CustomerMgr');
    var csrfProtection = require('dw/web/CSRFProtection');
    var CommonUtils = require('*/cartridge/scripts/common/limepayCommonUtils');

    var responseJson = {};
    if (!csrfProtection.validateRequest()) {
        CustomerMgr.logoutCustomer(false);
        responseJson.error = true;
        responseJson.redirectUrl = URLUtils.url('CSRF-Fail').toString();
        app.getView({
            JSONResponse: responseJson
        }).render('util/responsejson');
        return;
    }

    var currentBasket = BasketMgr.getCurrentBasket();
    if (!currentBasket) {
        responseJson.redirectUrl = URLUtils.url('Cart-Show').toString();
        responseJson.error = true;
        app.getView({
            JSONResponse: responseJson
        }).render('util/responsejson');
        return;
    }
    var token = request.httpParameterMap.paymentToken.stringValue;
    var phone = request.httpParameterMap.phone.stringValue;
    try {
        Transaction.wrap(function () {
            // Save token and formatted phone number to basket
            // used for create order service call
            if (!empty(token)) {
                currentBasket.custom.limepayPaymentToken = token;
            } else {
                throw new Error('Limepay token is missing');
            }
            if (!empty(phone)) {
                currentBasket.custom.limepayFormattedPhoneNo = phone;
            } else {
                throw new Error('Limepay phoneNo is missing');
            }
        });
    } catch (error) {
        CommonUtils.logError(error);
        responseJson.error = true;
        app.getView({
            JSONResponse: responseJson
        }).render('util/responsejson');
        return;
    }
    responseJson.success = true;

    app.getView({
        JSONResponse: responseJson
    }).render('util/responsejson');
}

/**
 * Remember Limepay toggle widget change in default selection
 * onto user session. Toggle widget to stay preselected with this state across site
 */
function saveSelection() {
    var userSelection = request.httpParameterMap.limepayToggle.stringValue;
    session.privacy.limepayToggleSelection = userSelection;
    var response = {
        success: true
    };

    app.getView({
        JSONResponse: response
    }).render('util/responsejson');
}

/**
 * Remote include to render dynamic Limepay user toggle widget
 */
function userToggleWidget() {
    var params = request.httpParameterMap;
    var limepayUtilsHelpers = require('*/cartridge/scripts/helpers/limepayUtilsHelpers');
    var limepayToggleSelection = limepayUtilsHelpers.getWidgetTogglePreselection();

    app.getView({
        limepayToggleSelection: limepayToggleSelection,
        currency: session.currency.currencyCode,
        amount: parseFloat(params.amount)
    }).render('/limepay/limepayusertogglewidget');
}

/**
 * Process the LimePay redirect after 3DS verification
 */
function limepay3DS() {
    var OrderMgr = require('dw/order/OrderMgr');
    var DWOrder = require('dw/order/Order');
    var Resource = require('dw/web/Resource');

    let Response = require('*/cartridge/scripts/util/Response');

    var params = request.httpParameterMap;
    var orderNo = params.OrderNo ? params.OrderNo.stringValue : '';
    var order = OrderMgr.getOrder(orderNo);

    if (!order) {
        return response.redirect(URLUtils.url('Cart-Show'));
    }

    var orderPaymentStatus = order.getPaymentStatus();

    if (orderPaymentStatus && orderPaymentStatus.value === DWOrder.PAYMENT_STATUS_PAID) {
        return response.redirect(URLUtils.url('Cart-Show'));
    }


    var handlePaymentsResult = handlePayments(order);

    // Limepay, return response error to checkout page - START
    if (handlePaymentsResult.error && handlePaymentsResult.PlaceOrderError) {
        return Transaction.wrap(function () {
            OrderMgr.failOrder(order);
            Response.renderJSON({
                error: true,
                PlaceOrderError: handlePaymentsResult.PlaceOrderError.message
            });

            return;
        });
    }
    // Limepay, return response error to checkout page - END

    if (handlePaymentsResult.error) {
        return Transaction.wrap(function () {
            OrderMgr.failOrder(order);
            Response.renderJSON({
                error: true,
                PlaceOrderError: Resource.msg('confirm.error.technical', 'checkout', null)
            });

            return;
        });

    } else if (handlePaymentsResult.missingPaymentInfo) {
        return Transaction.wrap(function () {
            OrderMgr.failOrder(order);
            Response.renderJSON({
                error: true,
                PlaceOrderError: Resource.msg('confirm.error.technical', 'checkout', null)
            });

            return;
        });
    } else {
        //Return LimePay response
        var responseObj = handlePaymentsResult.paymentResult || {};

        if (!(responseObj && responseObj.paymentActionRequired)) {
            var orderPlacementStatus = Order.submit(order);

            if (orderPlacementStatus.error) {
                Response.renderJSON({
                    error: true,
                    PlaceOrderError: Resource.msg('confirm.error.technical', 'checkout', null)
                });

                return;
            }

            clearForms();
            responseObj.submitOrder = true;
            responseObj.orderNo = orderNo;
            Response.renderJSON(responseObj);

            return;
        }

        Response.renderJSON(handlePaymentsResult);

        return;
    }
}

/**
 * Responsible for payment handling. This function uses PaymentProcessorModel methods to
 * handle payment processing specific to each payment instrument. It returns an
 * error if any of the authorizations failed or a payment
 * instrument is of an unknown payment method. If a payment method has no
 * payment processor assigned, the payment is accepted as authorized.
 *
 * @transactional
 * @param {dw.order.Order} order - the order to handle payments for.
 * @return {Object} JSON object containing information about missing payments, errors, or an empty object if the function is successful.
 */
function handlePayments(order) {
    var PaymentProcessor = app.getModel('PaymentProcessor');
    var PaymentMgr = require('dw/order/PaymentMgr');

    if (order.getTotalNetPrice().value !== 0.00) {

        var paymentInstruments = order.getPaymentInstruments();

        if (paymentInstruments.length === 0) {
            return {
                missingPaymentInfo: true
            };
        }
        /**
         * Sets the transaction ID for the payment instrument.
         */
        var handlePaymentTransaction = function () {
            paymentInstrument.getPaymentTransaction().setTransactionID(order.getOrderNo());
        };

        for (var i = 0; i < paymentInstruments.length; i++) {
            var paymentInstrument = paymentInstruments[i];

            if (PaymentMgr.getPaymentMethod(paymentInstrument.getPaymentMethod()).getPaymentProcessor() === null) {

                Transaction.wrap(handlePaymentTransaction);

            } else {

                var authorizationResult = PaymentProcessor.authorize(order, paymentInstrument);

                // Limepay, return response error to checkout page - START
                if (authorizationResult.error && authorizationResult.PlaceOrderError) {
                    return {
                        error: true,
                        PlaceOrderError: authorizationResult.PlaceOrderError
                    };
                }
                // Limepay, return response error to checkout page - END

                if (authorizationResult.not_supported || authorizationResult.error) {
                    return {
                        error: true
                    };
                }

                 //Add LimePay 3DS if 3DS redirect is enabled
                var paymentResult = authorizationResult.paymentResult;

                if (paymentResult && paymentResult.threeDSRedirect) {
                    return {
                        paymentResult: paymentResult.object,
                        threeDSURL: URLUtils.url('Limepay-Limepay3DS', 'threeDSCallback', true, 'OrderNo', order.orderNo).toString()
                    };
                }
            }
        }
    }

    return {};
}

function clearForms() {
    // Clears all forms used in the checkout process.
    session.forms.singleshipping.clearFormElement();
    session.forms.multishipping.clearFormElement();
    session.forms.billing.clearFormElement();
}

/* Web exposed methods */

/** Get the Basket context data
 * @see {@link module:controllers/LimePay~basketContext} */
exports.BasketContext = guard.ensure(['get'], basketContext);

/** Save Limepay transaction token
 * @see {@link module:controllers/LimePay~savePaymentToken} */
exports.SavePaymentToken = guard.ensure(['post', 'https'], savePaymentToken);

/** Save widget toggle selection onto user session
 * @see {@link module:controllers/LimePay~saveSelection} */
exports.SaveSelection = guard.ensure(['https'], saveSelection);

/** Renders Limepay customer dynamic toggle widget
 * @see {@link module:controllers/LimePay~userToggleWidget} */
exports.UserToggleWidget = guard.ensure(['https'], userToggleWidget);

/** Limepay 3DS entry after 3DS verification from Limepay 
 * @see {@link module:controllers/LimePay~limepay3DS} */
exports.Limepay3DS = guard.ensure(['https'], limepay3DS)
