/* API Includes */
var Transaction = require('dw/system/Transaction');
var Status = require('dw/system/Status');
var Resource = require('dw/web/Resource');

/* Script Modules */
var CommonUtils = require('*/cartridge/scripts/common/limepayCommonUtils');

const limepayPaymentMethodID = 'Limepay';

/**
 * Creates a Limepay payment instrument for the basket
 */
function Handle(args) {
    const currentBasket = args.Basket;
    var CheckoutHelpers = require('*/cartridge/scripts/helpers/limepayCheckoutHelpers');
    var PaymentMgr = require('dw/order/PaymentMgr');

    try {
        Transaction.wrap(function () {
            const result = CheckoutHelpers.removePaymentInstruments(currentBasket);

            if (result.error) {
                CommonUtils.logError(Resource.msg('limepay.remove.payinstrument.failed', 'limepay', null));
                return result;
            }

            var paymentInstrument = currentBasket.createPaymentInstrument(
                limepayPaymentMethodID,
                CheckoutHelpers.getNonGiftCertificateAmount(currentBasket)
            );

            var paymentTransaction = paymentInstrument.getPaymentTransaction();
            var paymentProcessor = PaymentMgr.getPaymentMethod(limepayPaymentMethodID).getPaymentProcessor();
            paymentTransaction.setPaymentProcessor(paymentProcessor);
        });
        CommonUtils.logDebug('Limepay payment instrument successfully created');
    } catch (e) {
        var errorMsg = Resource.msg('limepay.create.payinstrument.failed', 'limepay', null) + ' : ' + e.message;
        CommonUtils.logError(errorMsg);

        return {
            success: false,
            error: true,
            errorMessage: e.message
        };
    }

    return { success: true };
}

/**
 *  Calls Limepay create order API, and subsequently executes payment authorisation
 */
function Authorize(args) {
    var order = args.Order;
    var paymentInstrument = args.PaymentInstrument;
    var LimepayOrderService = require('*/cartridge/scripts/services/limepayOrderService');
    var LimepayPaymentService = require('*/cartridge/scripts/services/limepayPaymentService');

    // Limepay create order API call
    const orderResult = LimepayOrderService.createLimepayOrder(order, paymentInstrument);

    if (!orderResult.ok) {
        // Failed to create order at Limepay, authorisation call cannot be proceeded
        var errorMessage = CommonUtils.getResponseErrorMsg(orderResult.errorMessage);
        CommonUtils.logError(Resource.msg('limepay.createorder.failed', 'limepay', null));
        return {
            error: true,
            PlaceOrderError: new Status(Status.ERROR, errorMessage.detail)
        };
    }

    // Limepay create payment API call
    const paymentResult = LimepayPaymentService.createPayment(order, paymentInstrument);

    if (!paymentResult.ok) {
        // Payment failed at Limepay
        var errorMessage = CommonUtils.getResponseErrorMsg(paymentResult.errorMessage);
        CommonUtils.logError(Resource.msg('limepay.payment.failed', 'limepay', null));
        return {
            error: true,
            paymentResult: paymentResult,
            PlaceOrderError: new Status(Status.ERROR, errorMessage.detail)
        };
    }

    CommonUtils.logDebug('Limepay payment authorisation completed successfully');

    return {
        authorized: true,
        error: false,
        paymentResult: paymentResult
    };
}

exports.Handle = Handle;
exports.Authorize = Authorize;
