'use strict';

var util = require('./util');

// E.164 phone number format google library
var libphonenumber = require('google-libphonenumber');

// Limepay global variables
var limepayPhone,
    formattedPhoneNo,
    limepayEmail;
var LimepayPayment = {
    LimepayCheckoutContext: null,
    LimepayOrderData: null,
    LimepayPaymentPlanTermsAccepted: false
};

LimepayPayment.init = function (setup) {
    this.LimepayCheckoutContext = setup.context;
    this.LimepayOrderData = setup.order;

    if (window.LimepayCheckout && window.LimepayCheckout !== undefined && isLimepayPaymentSelected()) {
        var $submitBillingFormButton = $('button[name=dwfrm_billing_save]');
        $submitBillingFormButton.addClass('hide');
        $('body').trigger('limePayment:handlePaymentTabEvents');
    }
};

/**
 * Checks if the current selected payment method is Limepay
 * @returns {Boolean}
 */
function isLimepayPaymentSelected() {
    return $('input[name$="_selectedPaymentMethodID"]:checked').val() === 'Limepay';
}

/**
 * Save the generated Limepay payment token to basket, for completing order
 * @param {String} token
 */
function savePaymentToken(token) {
    if (window.LimepayCheckout && window.LimepayCheckout !== undefined && token !== undefined) {
        var saveTokenURL = window.LimepayContext.savePaymentTokenURL;
        var csrfTokenName = $('#limepay-checkout').attr('data-token-name');
        var csrfTokenValue = $('#limepay-checkout').attr('data-token-value');
        saveTokenURL = saveTokenURL + '?' + csrfTokenName + '=' + csrfTokenValue;
        var data = {
            paymentToken: token,
            phone: formattedPhoneNo
        };
        $.ajax({
            type: 'post',
            async: false,
            dataType: 'json',
            data: data,
            url: saveTokenURL
        })
            .done(function (data) {
                if (data && data.error && data.redirectUrl) {
                    location.href = data.redirectUrl;
                } else {
                    $('body').trigger('limePayment:postSubmitLimePayTokenEvents');
                }
            })
            .fail(function (error) {
            })
            .always(function () {
            });
    }
}

/**
 * Initializes or re-renders Limepay checkout form
 * @param {Object} context - Limepay checkout form configuration
 * @param {Object} order - customer order data for Limepay checkout
 */
function initLimepayCheckout(context, order) {
    // change any other submit button with this extension before loading the widget
    $('body').trigger('limePayment:handleLimePaySubmitButton');

    limepayEmail = '';
    if ($('input[name$=_billingAddress_email_emailAddress]').length > 0 && $('input[name$=_billingAddress_email_emailAddress]').val() !== '') {
        limepayEmail = $('input[name$=_billingAddress_email_emailAddress]').val();
    }

    formattedPhoneNo = getFormattedPhoneNo();

    var billingDetails = getBillingDetails();
    var limepayPrimaryColor = getPrimaryColor();
    var limepayCustomerToken = getCustomerToken();

    LimepayCheckout.init({
        publicKey: context.publicKey,
        email: limepayEmail,
        phone: formattedPhoneNo,
        customerFirstName: billingDetails.firstName,
        customerLastName: billingDetails.lastName,
        customerResidentialAddress: billingDetails.customerResidentialAddress,
        customerToken: limepayCustomerToken,
        hidePayLaterOption: context.hidePayLaterOption,
        hideFullPayOption: context.hideFullPayOption,
        paymentType: context.preselectedPaymentType == 'full' ? 'paycard' : 'payplan',
        paymentToken: function (token) {
            savePaymentToken(token);
        }
    });
    LimepayCheckout.render({
        elementId: 'limepay-cont',
        currency: order.orderInfo.currency,
        amount: order.orderInfo.amount,
        paymentType: context.preselectedPaymentType == 'full' ? 'paycard' : 'payplan',
        showPayNow: false,
        showPayPlanSubmit: false,
        checkoutCSSOverride: context.checkoutCSSOverride,
        primaryColor: limepayPrimaryColor
    });
    LimepayCheckout.errorHandler(function (err) {
        // Handle errors
    });
    LimepayCheckout.eventHandler(function (event) {
        $('body').trigger('limePayment:handleLimePayEvents', { event: event });
    });
}

function handleLimepayCheckoutTab(context, order) {
    if (window.LimepayContext.isLimepayEnabled) {
        initLimepayCheckout(context, order);
    }
}

function updateLimepayOrderAmount() {
    $.ajax({
        type: 'GET',
        async: false,
        dataType: 'json',
        url: LimepayPayment.LimepayCheckoutContext.basketContextURL,
        success: function (data) {
            if (typeof data === 'object' && data.error) {
                location.href = data.redirectUrl;
                return;
            }
            LimepayPayment.LimepayOrderData = data;
            if (isLimepayPaymentSelected()) {
                LimepayCheckout.update({ amount: data.orderInfo.amount });
            }
        },
        failure: function () {
        }
    });
}

/**
 * Parses billing phone number & country to generate phone in E164 format
 * @returns {String} e164FormattedPhoneNo - phone number in E164 format
 */
function getFormattedPhoneNo() {
    // Billing form values
    limepayPhone = $('input[name$=_billing_billingAddress_addressFields_phone]').val();
    var billingCountryCode = $('select[name$=_billing_billingAddress_addressFields_country]').val();
    var e164FormattedPhoneNo = '';

    if (limepayPhone && billingCountryCode) {
        var PNF = libphonenumber.PhoneNumberFormat;
        var phoneUtil = libphonenumber.PhoneNumberUtil.getInstance();
        var number = phoneUtil.parseAndKeepRawInput(limepayPhone, billingCountryCode);
        e164FormattedPhoneNo = phoneUtil.format(number, PNF.E164);
    }

    return e164FormattedPhoneNo;
}

/**
 * Retrieve billing details for Limepay checkout rendering
 * @returns {Object} - customer billing details
 */
function getBillingDetails() {
    // Billing customer
    var firstName = $('input[name$=_billing_billingAddress_addressFields_firstName]').val();
    var lastName = $('input[name$=_billing_billingAddress_addressFields_lastName]').val();

    // Billing Address
    var address1 = $('input[name$=_billing_billingAddress_addressFields_address1]').val();
    var address2 = $('input[name$=_billing_billingAddress_addressFields_address2]').val();
    var city = $('input[name$=_billing_billingAddress_addressFields_city]').val();
    var country = $('select[name$=_billing_billingAddress_addressFields_country] option:selected').text();
    var stateCode = $('select[name$=_billing_billingAddress_addressFields_states_state]').val();
    var postalCode = $('input[name$=_billing_billingAddress_addressFields_postal]').val();

    var residentialAddress = '';
    if (address1) residentialAddress += address1 + ', ';
    if (address2) residentialAddress += address2 + ', ';
    if (city) residentialAddress += city + ' ';
    if (stateCode) residentialAddress += stateCode + ' ';
    if (postalCode) residentialAddress += postalCode + ', ';
    if (country) residentialAddress += country;

    return {
        firstName: firstName,
        lastName: lastName,
        customerResidentialAddress: residentialAddress
    };
}

/**
 * Parses limepay primary color
 * @returns {String} limepayPrimaryColor - primary color
 */
function getPrimaryColor() {
    var limepayPrimaryColor = $('input[name$=limepay-primary-color]').val();
    return limepayPrimaryColor;
}


/**
 * Parses limepay customer token
 * @returns {String} limepayCustomerToken - customer token
 */
function getCustomerToken() {
    var limepayCustomerToken = $('input[name$=limepay-customer-token]').val();
    return limepayCustomerToken;
}

exports.init = function init() {
    window.LimepayContext = $('#limepay-checkout-context').data('checkoutcontext');
    window.LimepayOrderData = $('#limepay-order-data').data('orderdata');

    LimepayPayment.init({
        context: window.LimepayContext,
        order: window.LimepayOrderData
    });

    $('body').on('limePayment:handlePaymentTabEvents', function () {
        handleLimepayCheckoutTab(LimepayPayment.LimepayCheckoutContext, LimepayPayment.LimepayOrderData);
    });

    $('body').on('limePayment:handleLimePaySubmitButton', function () {
        var $submitBillingFormButton = $('button[name=dwfrm_billing_save]');

        var limepayPaymentBtn = $('.limepay-submit-payment');
        if ($(limepayPaymentBtn).length == 0) {
            var limePaySubmitBtn = $('<button/>', { 'class': 'button-fancy-large limepay-submit-payment', 'type': 'button' });
            limepayPaymentBtn = limePaySubmitBtn.insertAfter($submitBillingFormButton);
            limepayPaymentBtn.text($submitBillingFormButton.text());
            var checkoutContext = LimepayPayment.LimepayCheckoutContext;
            if ((checkoutContext.preferredPaymentType == 'splitOnly') ||
                (checkoutContext.preselectedPaymentType == 'split' && !checkoutContext.hidePayLaterOption && !checkoutContext.bnplCheckoutThresholdExceeds)) {
                limepayPaymentBtn.prop('disabled', true);
            }
        }
    });

    $('body').on('limePayment:handleLimePayEvents', function (e, data) {
        if (data.event && 'limepay_toggled_payment_type' == data.event.eventName) {
            LimepayPayment.LimepayPaymentPlan = data.event.eventData.paymentType;
            if ('paycard' === data.event.eventData.paymentType) {
                LimepayPayment.LimepayPaymentPlanTermsAccepted = false;
            }

            // Save toggle selection change to user session
            var userToggleSelection = data.event.eventData.paymentType == 'payplan' ? 'split' : 'full';
            var saveSelectionURL = LimepayPayment.LimepayCheckoutContext.saveToggleSelectionURL;
            saveSelectionURL = saveSelectionURL + '?limepayToggle=' + userToggleSelection;

            // Update context object selection
            LimepayPayment.LimepayCheckoutContext.preselectedPaymentType = userToggleSelection;

            $.ajax({
                url: saveSelectionURL,
                success: function (data) {
                }
            });
        }

        if (data.event && 'limepay_first_payment_amount_changed' == data.event.eventName && !LimepayPayment.LimepayPaymentPlan) {
            LimepayPayment.LimepayPaymentPlan = 'payplan';
            LimepayPayment.LimepayPaymentPlanTermsAccepted = false;
        }

        if (data.event && 'limepay_payment_plan_terms_accepted' == data.event.eventName) {
            LimepayPayment.LimepayPaymentPlan = 'payplan';
            LimepayPayment.LimepayPaymentPlanTermsAccepted = true;
        }

        // Update Payment Button Based on Payment Plans
        $('body').trigger('limePayment:handlePaymentSubmitBtn');
    });

    $('body').on('limePayment:handlePaymentSubmitBtn', function () {
        if (LimepayPayment.LimepayPaymentPlan && 'paycard' == LimepayPayment.LimepayPaymentPlan) {
            $('.limepay-submit-payment').prop('disabled', false);
        } else if (LimepayPayment.LimepayPaymentPlan && 'payplan' == LimepayPayment.LimepayPaymentPlan) {
            $('.limepay-submit-payment').prop('disabled', !LimepayPayment.LimepayPaymentPlanTermsAccepted);
        }
    });

    $('body').on('click', '.limepay-submit-payment', function (e) {
        e.preventDefault();
        if (isLimepayPaymentSelected() && null !== LimepayCheckout) {
            if ($('form[id$="_billing"]').valid()) {
                LimepayCheckout.submit();
            }
        }
    });

    $('body').on('limePayment:postSubmitLimePayTokenEvents', function () {
        // Trigger the regular place order button to proceed checkout
        $('button[name=dwfrm_billing_save]').trigger('click');
    });

    $('body').on('limePayment:updateLimePayOrderAmount', function () {
        var $billingForm = $('#dwfrm_billing');
        if ($billingForm.length > 0 && null !== LimepayCheckout) {
            updateLimepayOrderAmount();
        }
    });

    // Update to email or phone field, re-renders limepay checkout form
    var limepayDynamicFields = [
        'input[name$=_billing_billingAddress_email_emailAddress]',
        'input[name$=_billing_billingAddress_addressFields_phone]'
    ];
    limepayDynamicFields.forEach(function (selector) {
        $(selector).on('blur', function (e) {
            var entryUpdated = $(this).attr('name').indexOf('email') > -1 ? ($(this).val() != limepayEmail) : ($(this).val() != limepayPhone);
            if (isLimepayPaymentSelected() && entryUpdated) {
                initLimepayCheckout(LimepayPayment.LimepayCheckoutContext, LimepayPayment.LimepayOrderData);
            }
        });
    });
};

/**
 * Intiate the 3DS flow
 * @param {Object} threeDSData - 3DS object for performing actions
 */
exports.initLimePay3DS = function (threeDSData) {
    // Intialize 3DS flow, display action prompt
    if (window.LimepayCheckout && threeDSData && threeDSData.paymentResult.paymentActionRequired) {
        LimepayCheckout.handlePaymentActionRequired(
            threeDSData.paymentResult.paymentActionRequired, // from PayOrder HTTP 403 error response
            function () {
                $.ajax({
                    url: threeDSData.threeDSURL,
                    type: 'POST',
                    success: function (data) {
                        if (!data) {
                            window.location.href = URLs.cartShow;
                        } else {
                            var paymentResult = data.paymentResult || {};

                            if (paymentResult.paymentActionRequired) {
                                initLimePay3DS(data.paymentResult);
                            } else if (data.error || data.submitOrder) {
                                // redirect customer to billing page with error
                                var $orderForm = $('.submit-order');
                                var $orderSubmitURL = $orderForm.attr('action');
                                $orderSubmitURL = util.removeParamFromURL($orderSubmitURL, 'format');
                                $orderSubmitURL = util.appendParamsToUrl($orderSubmitURL, data);
                                $orderForm.attr('action', $orderSubmitURL);
                                window.submitOrder = true;
                                $orderForm.find('button[name="submit"]').click();
                            }
                        }
                    },
                    error: function (err) {
                        //redirect to error page
                        window.location.href = URLs.cartShow;
                    }
                });
              // action completed, reattempt PayOrder request
            },
            function (error) {
                window.location.href = URLs.cartShow;
            }
          );
    }
};